/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.ai;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.apache.tika.Tika;

import org.ametys.core.ui.Callable;
import org.ametys.core.upload.Upload;
import org.ametys.core.upload.UploadManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.ai.provider.AIProvider;
import org.ametys.plugins.ai.provider.AIProviderExtensionPoint;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.DeferredServiceable;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;


/**
 * This is helper to use AI as image or text generation.
 */
public class AIHelper extends AbstractLogEnabled implements Component, DeferredServiceable, Contextualizable
{
    /** Avalon role */
    public static final String ROLE = AIHelper.class.getName();
    /** The site configuration parameter for the a specific image prompt  */
    public static final String SITE_CONFIG_IMAGE_PROMPT = "aiImagePrompt";
    /** The site configuration parameter for the a specific text prompt  */
    public static final String SITE_CONFIG_TEXT_PROMPT = "aiTextPrompt";

    /** A security hardcoded max for the text summarize */
    private static final int __TEXT_SUMMARIZE_MAXLENGTH = 1000;
    /** The prompt used to summarize a text. It contains one parameter: {maxLength}. */
    private static final String __TEXT_SUMMARIZE_PROMPT_PREFIX =
            "Create a summary of the following user provided text in **{maxLength} characters maximum**. Do not exceed this limit."
                    + "The summary is to be used in a search results page. "
                    + "Provide the answer in the language of the text to be summarized. Not another language. "
                    + "Provide the answer without any formating, title, or quotes, just raw text. "
                    + "If the text is incomprehensible, or if it's just a few words: just return the same text.";

    /** The prompt to generate an image. It contains no parameter. */
    private static final String __IMAGE_GENERATION_PROMPT_PREFIX =
            "Generate an image/illustration/photo to illustrate the following user provided text. The image must have no text (if any it should be in the same language as the text)."; // No text in the image, since generated text are always unreadable.

    /** The activate param */
    private static final String CONFIG_ACTIVATE = "ai.active";
    /** The provider configuration param */
    private static final String CONFIG_PROVIDER = "ai.provider";
    
    /** The extension point providing access to registered AI providers.*/
    protected AIProviderExtensionPoint _aiProviderEP;
    
    /** The upload manager used to upload images */
    protected UploadManager _uploadManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    private SiteManager _siteManager;

    private Context _context;
    
    public void deferredService(ServiceManager manager) throws ServiceException
    {
        _aiProviderEP = (AIProviderExtensionPoint) manager.lookup(AIProviderExtensionPoint.ROLE);
        _uploadManager = (UploadManager) manager.lookup(UploadManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Checks if a configured AI provider is available and supports text summarization.
     * This includes verifying the provider exists, has a valid text model, and is properly configured.
     * @return true if all is good, false otherwise
     */
    public boolean isEnabled()
    {
        return _getCurrentAIProvider() != null;
    }
    
    /**
     * Summarize a given text
     * @param text The input text to be summarized.
     * @param maxLength The maximum number of characters for the summary.
     * @return A map with the Ai-generated summary under the key "summary".
     */
    @Callable(rights = "AI_Rights_Use")
    public Map<String, String> textToSummary(String text, int maxLength)
    {
        try
        {
            if (isEnabled())
            {
                return Map.of("summary", _getCurrentAIProvider().textToSummary(text, maxLength));
            }
        }
        catch (Exception e)
        {
            getLogger().error("Error while summarizing text", e);
        }
        return Map.of("error", "error");
    }
    
    /**
     * Checks if a configured AI provider is available and supports image generation.
     * This includes verifying the provider exists, has a valid text model, and is properly configured.
     * @return true if all is good, false otherwise
     */
    public boolean isImageGenerationSupported()
    {
        AIProvider aiProvider = _getCurrentAIProvider();
        if (aiProvider == null)
        {
            return false;
        }
        return aiProvider.isImageGenerationSupported();
    }

    /**
     * Generate an image with a given text
     * @param text The input text to generate the image with.
     * @param name The name for the generated image file.
     * @return A map with the URL of the Ai-generated image under the key "image".
     */
    @Callable(rights = "AI_Rights_Use")
    public Map<String, Object> textToImage(String text, String name)
    {
        try
        {
            if (isImageGenerationSupported())
            {
                Map<String, Object> imageAsJson = new HashMap<>();
                
                Path image = _getCurrentAIProvider().textToImage(text);
                try (InputStream is = Files.newInputStream(image))
                {
                    Upload storeUpload = _uploadManager.storeUpload(_currentUserProvider.getUser(), name + "-AI-generated." + StringUtils.substringAfterLast(image.toString(), "."), is);
                    _fillSuccess(storeUpload, imageAsJson);
                    return imageAsJson;
                }
                catch (Exception e)
                {
                    getLogger().error("Error while accessing to the image", e);
                    return Map.of("error", "error");
                }
                finally
                {
                    Files.delete(image);
                }
            }
        }
        catch (Exception e)
        {
            getLogger().error("Error while generating the image", e);
        }
        return Map.of("error", "error");
    }
    
    /**
     * Summarize a resource
     * @param resource the resource to be summarized.
     * @param maxLength The maximum number of characters for the summary.
     * @return the summary.
     * @throws Exception if an error occurred
     */
    public String resourceToSummary(Resource resource, int maxLength) throws Exception
    {
        if (isEnabled())
        {
            try (InputStream is = resource.getInputStream())
            {
                Tika tika = new Tika();
                tika.setMaxStringLength(-1);
                String text = tika.parseToString(is);
                
                return _getCurrentAIProvider().textToSummary(text, maxLength);
            }
        }
        return null;
    }
    
    /**
     * Fill the result map.
     * @param upload The upload
     * @param result The result map to fill
     */
    private void _fillSuccess(Upload upload, Map<String, Object> result)
    {
        result.put("success", true);
        result.put("id", upload.getId());
        result.put("filename", upload.getFilename());
        result.put("size", upload.getLength());
        result.put("viewHref", _getUrlForView(upload));
        result.put("downloadHref", _getUrlForDownload(upload));
    }
    
    /**
     * Get the url for view the uploaded file
     * @param upload The file uploaded
     * @return The url for view
     */
    private String _getUrlForView(Upload upload)
    {
        return "/plugins/core/upload/file?id=" + upload.getId();
    }
    
    /**
     * Get the url for download the uploaded file
     * @param upload The file uploaded
     * @return The url for view
     */
    private String _getUrlForDownload(Upload upload)
    {
        return "/plugins/core/upload/file?id=" + upload.getId() + "&download=true";
    }
    /**
     * Get the current AI Provider class
     * @return the AI provider
     */
    protected AIProvider _getCurrentAIProvider()
    {
        if (Config.getInstance().getValue(CONFIG_ACTIVATE) == Boolean.TRUE)
        {
            String providerID = Config.getInstance().getValue(CONFIG_PROVIDER);
            return _aiProviderEP.getExtension(providerID);
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Initialize the prompt with a given maximum length minus 10 characters to prevent the LLM to exceed the limit.
     * @param maxLength the number of characters allowed
     * @return the text of the prompt
     */
    public String getTextSummaryPrompt(int maxLength)
    {
        String siteName = null;
        
        Request request = ContextHelper.getRequest(_context);
        if (request != null)
        {
            siteName = WebHelper.getSiteName(request);
        }
        
        String prompt = "";
        
        if (StringUtils.isNotBlank(siteName))
        {
            try
            {
                Site site = _siteManager.getSite(siteName);
                prompt = site.getValue(SITE_CONFIG_TEXT_PROMPT);
                Strings.CS.appendIfMissing(prompt, ".");
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn("There is not site '{}' to generate an image prompt", siteName);
            }
            
        }

        return (__TEXT_SUMMARIZE_PROMPT_PREFIX + " " + prompt).replace("{maxLength}", Integer.toString(Math.min(maxLength, __TEXT_SUMMARIZE_MAXLENGTH)));
    }
    

    /**
     * Initialize the prompt for image generation.
     * @return the text of the prompt
     */
    public String getImageGenerationPrompt()
    {
        String siteName = null;
        
        Request request = ContextHelper.getRequest(_context);
        if (request != null)
        {
            siteName = WebHelper.getSiteName(request);
        }
        
        String prompt = "";
        
        if (StringUtils.isNotBlank(siteName))
        {
            try
            {
                Site site = _siteManager.getSite(siteName);
                prompt = site.getValue(SITE_CONFIG_IMAGE_PROMPT);
                Strings.CS.appendIfMissing(prompt, ".");
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn("There is not site '{}' to generate an image prompt", siteName);
            }
            
        }
        return __IMAGE_GENERATION_PROMPT_PREFIX + " " + prompt;
    }
    
}
