/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ai.provider;

import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;

/**
 * {@link Enumerator} for the AI enumerator.
 */
public class AIProviderEnumerator implements Enumerator<String>, Serviceable, Component
{
    /** Avalon role */
    public static final String ROLE = AIProviderEnumerator.class.getName();
    
    /** The AI component extension point */
    protected AIProviderExtensionPoint _aiComponentEP;
    
    /** The service manager */
    protected ServiceManager _manager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
    }
    
    private AIProviderExtensionPoint _getAIProviderExtensionPoint()
    {
        if (_aiComponentEP == null)
        {
            try
            {
                _aiComponentEP = (AIProviderExtensionPoint) _manager.lookup(AIProviderExtensionPoint.ROLE);
            }
            catch (ServiceException e)
            {
                // Can be called during application startup
                return null;
            }
        }
        return _aiComponentEP;
    }
    
    @Override
    public I18nizableText getEntry(String value) throws Exception
    {
        AIProviderExtensionPoint aiProviderEP = _getAIProviderExtensionPoint();
        if (aiProviderEP != null && aiProviderEP.hasExtension(value))
        {
            return aiProviderEP.getExtension(value).getLabel();
        }
        else
        {
            // Can be called during application startup
            return new I18nizableText(value);
        }
    }

    @Override
    public Map<String, I18nizableText> getEntries() throws Exception
    {
        AIProviderExtensionPoint aiProviderEP = _getAIProviderExtensionPoint();
        if (aiProviderEP == null)
        {
            return Map.of();
        }
        
        return aiProviderEP.getExtensionsIds()
            .stream()
            .map(id -> aiProviderEP.getExtension(id))
            .filter(Objects::nonNull)
            .collect(
                Collectors.toMap(
                    AIProvider::getId,
                    AIProvider::getLabel
                )
            );
    }
}
