/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ai.provider;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.ai.provider.impl.AbstractAIProvider;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.plugin.component.DeferredServiceable;

/**
 * Enumerates possible models depending on chosen aiprovider
 */
public class AITextModelEnumerator implements Enumerator<String>, DeferredServiceable, Component
{
    /** Avalon role */
    public static final String ROLE = AITextModelEnumerator.class.getName();
    
    private AIProviderExtensionPoint _aiProviderExtensionPoint;
    
    public void deferredService(ServiceManager manager) throws ServiceException
    {
        _aiProviderExtensionPoint = (AIProviderExtensionPoint) manager.lookup(AIProviderExtensionPoint.ROLE);
    }
    
    public Map<String, I18nizableText> getEntries() throws Exception
    {
        throw new UnsupportedOperationException(ROLE + " must be configured with one dependency");
    }
    
    public Map<String, I18nizableText> getEntries(Map<String, Object> contextualParameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        List<String> dependencies = (List<String>) contextualParameters.get("dependencies");
        
        if (dependencies.size() != 1)
        {
            throw new IllegalArgumentException(ROLE + " must bu configured with one dependency");
        }
        
        String aiProviderId = dependencies.get(0);
        if (StringUtils.isNotBlank(aiProviderId)
            && _aiProviderExtensionPoint.hasExtension(aiProviderId))
        {
            AIProvider aiProvider = _aiProviderExtensionPoint.getExtension(aiProviderId);
            if (aiProvider instanceof AbstractAIProvider absAIProvider)
            {
                return absAIProvider.getKnownTextModels().stream().collect(Collectors.toMap(v -> v, v -> new I18nizableText(v)));
            }
        }
        return Map.of();
    }
}
