/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.events;

import java.io.IOException;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.filter.ContentFilterHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.type.ModelItemType;
import org.ametys.runtime.model.type.ModelItemTypeConstants;
import org.ametys.web.filter.WebContentFilter;
import org.ametys.web.filter.WebContentFilterHelper;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;

/**
 * Abstract generator provinding methods to sax an event content
 */
public abstract class AbstractEventGenerator extends ServiceableGenerator
{
    /** The source resolver. */
    protected SourceResolver _resolver;
    
    /** The content filter helper. */
    protected WebContentFilterHelper _filterHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _filterHelper = (WebContentFilterHelper) serviceManager.lookup(ContentFilterHelper.ROLE);
    }
    
    /**
     * SAX a content
     * 
     * @param handler The content handler to SAX into
     * @param content The content.
     * @param saxContentItSelf true to sax the content, false will only sax some meta
     * @param filter The filter. Can be null if saxContentItSelf is false
     * @param checkUserAccess True to check user access when saxing the content itself
     * @throws SAXException If an error occurs while SAXing
     * @throws IOException If an error occurs while retrieving content.
     */
    public void saxContent(ContentHandler handler, Content content, boolean saxContentItSelf, WebContentFilter filter, boolean checkUserAccess) throws SAXException, IOException
    {
        List<String> params = new ArrayList<>();
        params.add(content.getTitle());

        AttributesImpl attrs = new AttributesImpl();
        
        String start = _getFormatedDateAttribute(content, EventsFilterHelper.START_DATE_META);
        if (start != null)
        {
            params.add(start);
            attrs.addCDATAAttribute("start", start);
        }
        
        String end = _getFormatedDateAttribute(content, EventsFilterHelper.END_DATE_META);
        if (end != null)
        {
            params.add(end);
            attrs.addCDATAAttribute("end", end);
        }

        XMLUtils.startElement(handler, "event", attrs);

        String key = end == null ? "CALENDAR_SERVICE_AGENDA_EVENT_TITLE_SINGLE_DAY" : "CALENDAR_SERVICE_AGENDA_FROM_TO";
        I18nizableText description = new I18nizableText(null, key, params);

        description.toSAX(handler, "description");

        if (saxContentItSelf)
        {
            // Link view.
            XMLUtils.startElement(handler, "view");
            _filterHelper.saxContent(handler, content, filter.getView(), checkUserAccess);
            XMLUtils.endElement(handler, "view");
        }

        // XML full view.
        saxXMLContent(handler, content, "main");

        if (content instanceof WebContent)
        {
            WebContent webContent = (WebContent) content;

            XMLUtils.startElement(handler, "pages");
            Collection<Page> pages = webContent.getReferencingPages();
            for (Page page : pages)
            {
                AttributesImpl atts = new AttributesImpl();
                atts.addCDATAAttribute("id", page.getId());
                atts.addCDATAAttribute("name", page.getName());
                atts.addCDATAAttribute("lang", page.getSitemapName());
                atts.addCDATAAttribute("site", page.getSiteName());
                atts.addCDATAAttribute("path", page.getPathInSitemap());
                atts.addCDATAAttribute("title", page.getTitle());
                XMLUtils.createElement(handler, "page", atts);
            }
            XMLUtils.endElement(handler, "pages");
        }
        XMLUtils.endElement(handler, "event");
    }

    /**
     * Retrieves the formated value of the date attribute of the given content
     * @param content the content
     * @param attributePath the path of the attribute. The attribute must be date or date time
     * @return the formated value
     */
    protected String _getFormatedDateAttribute(Content content, String attributePath)
    {
        ModelItemType attributeType = content.getType(attributePath);

        if (ModelItemTypeConstants.DATE_TYPE_ID.equals(attributeType.getId()))
        {
            LocalDate date = content.getValue(attributePath);
            return Optional.ofNullable(date).map(d -> d.format(DateTimeFormatter.ISO_LOCAL_DATE)).orElse(null);
        }
        else
        {
            ZonedDateTime date = content.getValue(attributePath);
            return Optional.ofNullable(date).map(d -> d.withZoneSameInstant(ZoneId.systemDefault()).format(DateTimeFormatter.ISO_LOCAL_DATE)).orElse(null);
        }
    }

    /**
     * SAX a content in XML mode.
     * 
     * @param handler The content handler to SAX into
     * @param content The content to SAX
     * @param viewName The view to use
     * @throws SAXException If an error occurs while SAXing
     * @throws IOException If an error occurs while retrieving content.
     */
    public void saxXMLContent(ContentHandler handler, Content content, String viewName) throws SAXException, IOException
    {
        String uri = "cocoon://_content.xml?contentId=" + content.getId() + "&viewName=" + viewName;
        SitemapSource src = null;

        try
        {
            src = (SitemapSource) _resolver.resolveURI(uri);
            src.toSAX(new IgnoreRootHandler(handler));
        }
        finally
        {
            _resolver.release(src);
        }
    }
}
