/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.events;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.filter.ContentFilter;
import org.ametys.cms.filter.ContentFilterExtensionPoint;
import org.ametys.cms.filter.ContentFilterHelper;
import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.calendar.events.EventsFilter.EventFilterSearchContext;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.DateExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.web.filter.WebContentFilter.AccessLimitation;
import org.ametys.web.filter.WebContentFilter.Context;
import org.ametys.web.filter.WebContentFilter.FilterSearchContext;
import org.ametys.web.filter.WebContentFilterHelper;
import org.ametys.web.frontoffice.search.instance.model.SiteContextType;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Helper for events filter
 *
 */
public class EventsFilterHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** The avalon role */
    public static final String ROLE = EventsFilterHelper.class.getName();

    /** The start date metadata. */
    public static final String START_DATE_META = "start-date";

    /** The end date metadata. */
    public static final String END_DATE_META = "end-date";
    
    /** The events filter ID. */
    public static final String EVENTS_FILTER_ID = "events";

    
    /** The tag provider extension point. */
    protected TagProviderExtensionPoint _tagProviderEP;
    /** Extension point for content filters */
    protected ContentFilterExtensionPoint _filterExtPt;
    /** The helper for static content filters */
    protected WebContentFilterHelper _contentFilterHelper;
    /** The JSON utils */
    protected JSONUtils _jsonUtils;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _tagProviderEP = (TagProviderExtensionPoint) manager.lookup(TagProviderExtensionPoint.ROLE);
        _filterExtPt = (ContentFilterExtensionPoint) manager.lookup(ContentFilterExtensionPoint.ROLE);
        _contentFilterHelper = (WebContentFilterHelper) manager.lookup(ContentFilterHelper.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }

    /**
     * Create a events filter from the static "events" filter
     * @param id the id of filter to create
     * @return the created events filter
     */
    public EventsFilter createEventsFilter(String id)
    {
        EventsFilter eventsFilter = (EventsFilter) _filterExtPt.getExtension(EventsFilterHelper.EVENTS_FILTER_ID);
        return new EventsFilter(id, eventsFilter, _contentFilterHelper);
    }
    
    /**
     * Get the service title.
     * 
     * @param zoneItem the zone item.
     * @return the service title.
     */
    public String getTitle(ZoneItem zoneItem)
    {
        return zoneItem.getServiceParameters().getValue("title", false, "");
    }

    /**
     * Get the link.
     * 
     * @param zoneItem the zone item.
     * @return the link.
     */
    public String getLink(ZoneItem zoneItem)
    {
        return zoneItem.getServiceParameters().getValue("link", false, "");
    }

    /**
     * Get the link title.
     * 
     * @param zoneItem the zone item.
     * @return the link title.
     */
    public String getLinkTitle(ZoneItem zoneItem)
    {
        return zoneItem.getServiceParameters().getValue("link-title", false, "");
    }

    /**
     * Get the default range type parameter value.
     * 
     * @param zoneItem the zone item.
     * @return the default range type.
     */
    public String getDefaultRangeType(ZoneItem zoneItem)
    {
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        // First check that there is a definition because calendar service doesn't define the default-range parameter
        return serviceParameters.hasDefinition("default-range") ? serviceParameters.getValue("default-range", false, StringUtils.EMPTY) : StringUtils.EMPTY;
    }

    /**
     * Mask orphan?
     * 
     * @param zoneItem the zone item.
     * @return mask orphan.
     */
    public boolean getMaskOrphan(ZoneItem zoneItem)
    {
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        // First check that there is a definition because calendar service doesn't define the mask-orphan parameter
        return serviceParameters.hasDefinition("mask-orphan") ? serviceParameters.getValue("mask-orphan", false, false) : false;
    }

    /**
     * Get the access limitation policy.
     * @param zoneItem the zone item.
     * @return the access limitation policy.
     */
    public AccessLimitation getAccessLimitation(ZoneItem zoneItem)
    {
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        // First check that there is a definition because calendar service doesn't define the handle-user-access parameter
        if (serviceParameters.hasDefinition("handle-user-access"))
        {
            boolean handleUserAccess = serviceParameters.getValue("handle-user-access", false, false);
            return handleUserAccess ? AccessLimitation.USER_ACCESS : AccessLimitation.PAGE_ACCESS;
        }
        else
        {
            return AccessLimitation.PAGE_ACCESS;
        }
    }
    
    /**
     * Is there a PDF download link on the service?
     * 
     * @param zoneItem the zone item.
     * @return true if a PDF download link will be present, false otherwise.
     */
    protected boolean getPdfDownload(ZoneItem zoneItem)
    {
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        // First check that there is a definition because calendar service doesn't define the pdf-download parameter
        return serviceParameters.hasDefinition("pdf-download") ? serviceParameters.getValue("pdf-download", false, false) : false;
    }

    /**
     * Is there an iCalendar export link on the service?
     * 
     * @param zoneItem the zone item.
     * @return true if an iCalendar export link will be present, false
     *         otherwise.
     */
    protected boolean getIcalDownload(ZoneItem zoneItem)
    {
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        // First check that there is a definition because calendar service doesn't define the ical-download parameter
        return serviceParameters.hasDefinition("ical-download") ? serviceParameters.getValue("ical-download", false, false) : false;
    }
    
    /**
     * Get the content types through the service parameters.
     * 
     * @param zoneItem the zone item.
     * @return the content types.
     */
    public Set<String> getContentTypes(ZoneItem zoneItem)
    {
        Set<String> contentTypes = new LinkedHashSet<>();
        
        String[] cTypes = zoneItem.getServiceParameters().getValue("content-types", false, ArrayUtils.EMPTY_STRING_ARRAY);
        if (cTypes.length > 0 && cTypes[0].length() > 0)
        {
            for (String cType : cTypes)
            {
                if (!"".equals(cType))
                {
                    contentTypes.add(cType);
                }
            }
        }

        return contentTypes;
    }

    /**
     * Get the all tags from the search contexts
     * @param zoneItem the zone item.
     * @param searchContexts the search contexts
     * @return the tags.
     */
    public Set<String> getTags(ZoneItem zoneItem, List<Map<String, Object>> searchContexts)
    {
        HashSet<String> hashSet = new HashSet<>();
        if (searchContexts == null)
        {
            // get search contexts from zoneitem
            ModifiableModelAwareRepeater search = zoneItem.getServiceParameters().getValue("search");
            for (ModifiableModelAwareRepeaterEntry repeaterEntry : search.getEntries())
            {
                hashSet.addAll(Arrays.asList(repeaterEntry.getValue("tags", false, ArrayUtils.EMPTY_STRING_ARRAY)));
            }
        }
        else
        {
            for (Map<String, Object> entry : searchContexts)
            {
                hashSet.addAll(Arrays.asList((String[]) entry.get("tags")));
            }
        }

        return hashSet;
    }
    
    /**
     * Get the all selected tags to be used as categories from the search contexts
     * @param zoneItem the zone item.
     * @param searchContexts the search contexts
     * @param currentSiteName the current site name.
     * @return the tags categories.
     */
    public Set<Tag> getTagCategories(ZoneItem zoneItem, List<Map<String, Object>> searchContexts, String currentSiteName)
    {
        Set<Tag> categories = new LinkedHashSet<>();
        List<Map<String, Object>> search = searchContexts;

        if (search == null)
        {
            // if null, get search contexts from zoneitem (only need sites and tag categories values)
            search = new ArrayList<>();
            ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();

            HashSet<String> hashSet = new HashSet<>();
            ModifiableModelAwareRepeater searchRepeater = serviceParameters.getValue("search");
            
            for (ModifiableModelAwareRepeaterEntry repeaterEntry : searchRepeater.getEntries())
            {
                Map<String, Object> mapEntry = new HashMap<>();
                // First check that there is a definition because calendar service doesn't define the tag-categories parameter
                if (repeaterEntry.hasDefinition("tag-categories"))
                {
                    mapEntry.put("tag-categories", repeaterEntry.getValue("tag-categories", false, ArrayUtils.EMPTY_STRING_ARRAY));
                    hashSet.addAll(Arrays.asList(repeaterEntry.getValue("tag-categories", false, ArrayUtils.EMPTY_STRING_ARRAY)));
                }
                if (repeaterEntry.hasValue("sites"))
                {
                    mapEntry.put("sites", repeaterEntry.getValue("sites", false, ArrayUtils.EMPTY_STRING_ARRAY));
                }
                search.add(mapEntry);
            }
            
        }
        
        
        search.forEach(searchContext ->
        {
            if (searchContext.containsKey("tag-categories"))
            {
                String[] categoriesArray = (String[]) searchContext.get("tag-categories");
                for (int i = 0; i < categoriesArray.length; i++)
                {
                    Map<String, Object> sitesData = _jsonUtils.convertJsonToMap((String) searchContext.get("sites"));
                    
                    @SuppressWarnings("unchecked")
                    List<String> sites = (List<String>) sitesData.get("sites");
                    String context = (String) sitesData.get("context");
                    SiteContextType siteContextType = SiteContextType.fromClientSideName(context);

                    Map<String, Object> tagParameters = new HashMap<>();
                    if (SiteContextType.CURRENT.equals(siteContextType))
                    {
                        tagParameters.put("siteName", currentSiteName);
                    }
                    else if (SiteContextType.AMONG.equals(siteContextType) && sites.size() == 1)
                    {
                        tagParameters.put("siteName", sites.get(0));
                    }
                    else
                    {
                        tagParameters.put("siteName", null);
                    }
                    
                    Tag category = _tagProviderEP.getTag(categoriesArray[i], tagParameters);
                    if (category != null)
                    {
                        categories.add(category);
                    }
                    
                }
            }
        });
        

        return categories;
    }

    /**
     * Get all the descendant tags belonging to a collection of input tags.
     * 
     * @param tags the collection of input tag.
     * @return an exhaustive set of the tags.
     */
    public Set<String> getAllTags(Collection<? extends Tag> tags)
    {
        Set<String> allTags = new LinkedHashSet<>();

        for (Tag tag : tags)
        {
            Map<String, ? extends Tag> childTagsMap = tag.getTags();

            if (childTagsMap != null)
            {
                Collection<? extends Tag> childTags = childTagsMap.values();
                for (Tag child : childTags)
                {
                    allTags.add(child.getName());
                }
                
                // recursive add.
                allTags.addAll(getAllTags(childTags));
            }
        }
        
        return allTags;
    }

    /**
     * Get all the descendant tags belonging to a single tag.
     * 
     * @param tag the tag.
     * @return an exhaustive set of the tags.
     */
    protected Set<Tag> getAllTags(Tag tag)
    {
        Set<Tag> allTags = new LinkedHashSet<>();

        Map<String, ? extends Tag> childTagsMap = tag.getTags();

        if (childTagsMap != null)
        {
            Collection<? extends Tag> childTags = childTagsMap.values();
            allTags.addAll(childTags);
            
            for (Tag child : childTags)
            {
                allTags.addAll(getAllTags(child));
            }
        }

        return allTags;
    }

    /**
     * Get the date range from the calendar type and parameters.
     * 
     * @param type the calendar mode: "calendar", "single-day" or "agenda".
     * @param year the year.
     * @param month the month.
     * @param day the day.
     * @param monthsBefore get x months before.
     * @param monthsAfter get x months after.
     * @param rangeType the range type, "month" or "week".
     * @return the date range.
     */
    public DateTimeRange getDateRange(String type, int year, int month, int day, int monthsBefore, int monthsAfter, String rangeType)
    {
        DateTimeRange dateRange = null;

        // Single day mode.
        if ("single-day".equals(type))
        {
            LocalDate date = _getDate(year, month, day);
            ZonedDateTime datetime = date.atStartOfDay(ZoneId.systemDefault());
            dateRange = new DateTimeRange(datetime, datetime.plusDays(1));
        }
        // JS calendar mode.
        else if ("calendar".equals(type) || "agenda".equals(type))
        {
            dateRange = _getDateRange(monthsBefore, monthsAfter);
        }
        else if ("period".equals(type))
        {
            LocalDate date = _getDate(year, month, day);
            dateRange = _getDateRange(date, monthsBefore, monthsAfter);
        }

        if (getLogger().isInfoEnabled() && dateRange != null)
        {
            String start = dateRange.fromDate().format(DateTimeFormatter.ISO_LOCAL_DATE);
            String end = dateRange.untilDate().format(DateTimeFormatter.ISO_LOCAL_DATE);
            getLogger().info("Getting contents between " + start + " and " + end);
        }

        return dateRange;
    }
    
    /**
     * Get a date range from a period (x months before now and x months after).
     * 
     * @param monthsBefore the wanted number of months before the current month.
     * @param monthsAfter the wanted number of months after the current month.
     * @return the date range.
     */
    protected DateTimeRange _getDateRange(int monthsBefore, int monthsAfter)
    {
        return _getDateRange(LocalDate.now(), monthsBefore, monthsAfter);
    }

    /**
     * Get a date range from a period, around a given date (x months before and
     * x months after the date).
     * 
     * @param localDate the date.
     * @param monthsBefore the wanted number of months before the current month.
     * @param monthsAfter the wanted number of months after the current month.
     * @return the date range.
     */
    protected DateTimeRange _getDateRange(LocalDate localDate, int monthsBefore, int monthsAfter)
    {
        LocalDate firstDayOfMonth = localDate.withDayOfMonth(1);
        
        LocalDate fromDate = firstDayOfMonth.minusMonths(monthsBefore);
        ZonedDateTime fromDateTime = fromDate.atStartOfDay(ZoneId.systemDefault());
        
        LocalDate untilDate = firstDayOfMonth.plusMonths(monthsAfter + 1);
        ZonedDateTime untilDateTime = untilDate.atStartOfDay(ZoneId.systemDefault());
        
        return new DateTimeRange(fromDateTime, untilDateTime);
    }

    /**
     * Get a date object from a year-month-day set.
     * 
     * @param year the year.
     * @param month the month (1 to 12).
     * @param day the day (1 to 31).
     * @return the date object.
     */
    protected LocalDate _getDate(int year, int month, int day)
    {
        LocalDate localDate = LocalDate.now();
        
        if (year > 0 && month > 0 && day > 0)
        {
            localDate.withYear(year);
            localDate.withMonth(month);
            localDate.withDayOfMonth(day);
        }

        return localDate;
    }

    /**
     * Get the metadata expression from the calendar type and date range.
     * 
     * @param type the calendar type, may be "calendar", "single-day",
     *            "agenda" or "period" (agenda = calendar = period).
     * @param dateRange the date range. Only the start date is used for
     *            "single-day" type.
     * @return the metadata Expression on contents.
     */
    public Expression getExpression(String type, DateTimeRange dateRange)
    {
        Expression expression = null;

        if (dateRange != null)
        {
            if ("single-day".equals(type))
            {
                expression = _getMetadataExpression(dateRange);
            }
            else if ("calendar".equals(type) || "agenda".equals(type) || "period".equals(type))
            {
                expression = _getMetadataExpression(dateRange);
            }
        }

        return expression;
    }

    /**
     * Get a metadata expression from a date range.
     * 
     * @param dateRange the date range.
     * @return the date range metadata expression.
     */
    protected Expression _getMetadataExpression(DateTimeRange dateRange)
    {
        if (dateRange != null)
        {
            // start-date < untilDate AND (end-date >= fromDate OR not(end-date) AND start-date >= fromDate)
            
            ZonedDateTime fromDate = dateRange.fromDate();
            ZonedDateTime untilDate = dateRange.untilDate();
            
            Expression startBeforeExpr = new DateExpression(START_DATE_META, Operator.LT, untilDate);
            Expression endAfterExpr = new DateExpression(END_DATE_META, Operator.GE, fromDate);
            Expression startAfterExpr = new DateExpression(START_DATE_META, Operator.GE, fromDate);
            Expression noEndExpr = new NotExpression(new MetadataExpression(END_DATE_META));
            
            return new AndExpression(startBeforeExpr, new OrExpression(endAfterExpr, new AndExpression(noEndExpr, startAfterExpr)));
        }

        return null;
    }

    /**
     * Get a metadata expression from a single date.
     * 
     * @param date the date.
     * @return the date metadata expression.
     */
    protected Expression _getMetadataExpression(Date date)
    {
        Expression expression = null;
        
        if (date != null)
        {
            Date nextDay = Date.from(date.toInstant().atZone(ZoneId.systemDefault()).plusDays(1).toInstant());
            
            Expression noEndDate = new NotExpression(new MetadataExpression(END_DATE_META));
            Expression startDate = new DateExpression(START_DATE_META, Operator.GE, date);
            Expression startDateNextDay = new DateExpression(START_DATE_META, Operator.LT, nextDay);
            Expression onlyStartDate = new AndExpression(noEndDate, startDate, startDateNextDay);
            
            Expression noStartDate = new NotExpression(new MetadataExpression(START_DATE_META));
            Expression endDate = new DateExpression(END_DATE_META, Operator.GE, date);
            Expression endDateNextDay = new DateExpression(END_DATE_META, Operator.LT, nextDay);
            Expression onlyEndDate = new AndExpression(noStartDate, endDate, endDateNextDay);
            
            Expression afterStart = new DateExpression(START_DATE_META, Operator.LT, nextDay);
            Expression beforeEnd = new DateExpression(END_DATE_META, Operator.GE, date);
            Expression bothDates = new AndExpression(afterStart, beforeEnd);
            
            expression = new OrExpression(onlyStartDate, onlyEndDate, bothDates);
        }
        
        return expression;
    }

    /**
     * Get last day of the week at a given date.
     * 
     * @param cal the date as a Calendar.
     * @return the last day of the week.
     */
    protected int getLastDayOfWeek(Calendar cal)
    {
        return Calendar.SUNDAY;
    }
    
    /**
     * Filter a list of categories depending on the service parameters (categoriesArray) and request tags (requestedCategoriesArray)
     * @param searchContexts the search contexts
     * @param requestedCategories list of requested categories (can be "all" to get all the categories from the service, or a list of categories)
     * @param zoneItem zone Item
     * @param siteName site name
     * @return a set of tags
     */
    public Set<String> getFilteredCategories(List<Map<String, Object>> searchContexts, String[] requestedCategories, ZoneItem zoneItem, String siteName)
    {
     // Select a single tag or "all".
        List<String> requestedTags = Arrays.asList(requestedCategories);
        // Get the categories of the tags to match, from the zone item or from
        // the parameters.
        Set<Tag> categoriesSet = getTagCategories(zoneItem, searchContexts, siteName);
        // Get all tags belonging to the categories.
        Set<String> allCategoriesNames = getAllTags(categoriesSet);
        // Restrain the list to the provided tags.
        Set<String> filteredCategory = new HashSet<>();
        if (requestedTags.contains("all"))
        {
            filteredCategory = allCategoriesNames;
        }
        else
        {
            filteredCategory = requestedTags.stream()
                    .map(String::toUpperCase)
                    .filter(allCategoriesNames::contains)
                    .collect(Collectors.toSet());
        }
        
        return filteredCategory;
    }
    
    /**
     * Generate the filter
     * @param dateRange range of dates to search
     * @param zoneItem used to get the tags
     * @param view metadataset used
     * @param type calendar, agenda, period (same thing) or "single-day" to get a single day
     * @param filteredCategories list of categories filtered by those requested by the user
     * @param searchContexts the search contexts
     * @return an EventFilter to match the parameters
     */
    public EventsFilter generateEventFilter(DateTimeRange dateRange, ZoneItem zoneItem, String view, String type, Set<String> filteredCategories, List<Map<String, Object>> searchContexts)
    {
        String zoneItemId = zoneItem.getId();
        
        Set<String> contentTypes = getContentTypes(zoneItem);
        boolean maskOrphan = getMaskOrphan(zoneItem);
        AccessLimitation accessLimitation = getAccessLimitation(zoneItem);

        EventsFilter eventsFilter = createEventsFilter(zoneItemId);

        Expression expression = getExpression(type, dateRange); // calendar, agenda, period do the same thing
        configureFilter(eventsFilter, expression, contentTypes, searchContexts, filteredCategories, view, maskOrphan, accessLimitation);
        
        return eventsFilter;
    }
    
    /**
     * Configure the filter to return the wanted contents.
     * 
     * @param eventsFilter the events filter.
     * @param expression the metadata Expression, can be null.
     * @param contentTypes a list of content types that will restrict the scope of the request.
     * @param searchContexts the input searchContext
     * @param orTags a list of tags among which only one is required for the
     *            content to match.
     * @param view the view.
     * @param maskOrphan true to prevent getting orphan contents.
     * @param accessLimitation The access limitation policy.
     */
    public void configureFilter(EventsFilter eventsFilter, Expression expression, Collection<String> contentTypes, List<Map<String, Object>> searchContexts, Set<String> orTags, String view, boolean maskOrphan, AccessLimitation accessLimitation)
    {
        // Set the content types expression, even if null
        if (eventsFilter.getContentTypes() != null)
        {
            eventsFilter.getContentTypes().clear();
        }
        
        for (String contentType : contentTypes)
        {
            eventsFilter.addContentType(contentType);
        }

        // Set the metadata expression, even if null.
        eventsFilter.setMetadataExpression(expression);

        if (StringUtils.isNotEmpty(view))
        {
            eventsFilter.setView(view);
        }

        eventsFilter.clearSearchContexts();
        _setSearchContext(eventsFilter, searchContexts, orTags);
        
        eventsFilter.setMaskOrphanContents(maskOrphan);
        
        eventsFilter.setAccessLimitation(accessLimitation);
    }

    /**
     * Set the search contexts in a filter from a service instance attributes.
     * @param zoneItem the service parameters data holder.
     * @param inputSearchContexts the input search contexts
     * @return a list of search context
     */
    public List<Map<String, Object>> getSearchContext(ZoneItem zoneItem, List<Map<String, Object>> inputSearchContexts)
    {
        List<Map<String, Object>> searchContexts = inputSearchContexts;

        if (searchContexts == null)
        {
            // if null, get search contexts from zoneitem
            searchContexts = new ArrayList<>();

            ModifiableModelAwareRepeater searchRepeater = zoneItem.getServiceParameters().getValue("search");
            
            for (ModifiableModelAwareRepeaterEntry repeaterEntry : searchRepeater.getEntries())
            {
                Map<String, Object> mapEntry = new HashMap<>();
                // First check that there is a definition because calendar service doesn't define the tag-categories parameter
                if (repeaterEntry.hasDefinition("tag-categories"))
                {
                    mapEntry.put("tag-categories", repeaterEntry.getValue("tag-categories", false, ArrayUtils.EMPTY_STRING_ARRAY));
                }

                mapEntry.put("sites", repeaterEntry.getValue("sites", false, ArrayUtils.EMPTY_STRING_ARRAY));
                mapEntry.put("tags", repeaterEntry.getValue("tags", false, ArrayUtils.EMPTY_STRING_ARRAY));
                mapEntry.put("search-context", repeaterEntry.getValue("search-context", false, ArrayUtils.EMPTY_STRING_ARRAY));
                mapEntry.put("context-lang", repeaterEntry.getValue("context-lang", false, StringUtils.EMPTY));
                mapEntry.put("strict-search-on-tags", repeaterEntry.getValue("strict-search-on-tags", false, false));
                
                searchContexts.add(mapEntry);
            }
            
        }
        return searchContexts;
    }
    
    /**
     * Get the search parameters from the search values.
     * @param filter the filter to configure.
     * @param searchContextValues the input searchContext
     * @param orTags a list of tags among which only one is required for the
     *            content to match.
     */
    protected void _setSearchContext(EventsFilter filter, List<Map<String, Object>> searchContextValues, Set<String> orTags)
    {
        List<Map<String, Object>> nonNullSearchContextValues = Optional.ofNullable(searchContextValues)
                                                                .orElse(Collections.EMPTY_LIST);
        
        for (Map<String, Object> entry : nonNullSearchContextValues)
        {
            FilterSearchContext filterContext = filter.addSearchContext();

            Map<String, Object> sitesData = _jsonUtils.convertJsonToMap((String) entry.get("sites"));
            String searchContext = (String) sitesData.get("context");
            
            Map<String, Object> searchContextMap = _jsonUtils.convertJsonToMap((String) entry.get("search-context"));
            String subSearchContext = (String) searchContextMap.get("context");
            if (StringUtils.isEmpty(searchContext)
                || subSearchContext != null && !Context.CURRENT_SITE.name().equals(subSearchContext))
            {
                // Delegate to the search-context enumeration context
                searchContext = subSearchContext;
            }
            
            if ("CHILD_PAGES".equals(searchContext) || "CHILD_PAGES_OF".equals(searchContext))
            {
                filterContext.setContext(Context.CHILD_PAGES);
            }
            else if ("DIRECT_CHILD_PAGES".equals(searchContext) || "DIRECT_CHILD_PAGES_OF".equals(searchContext))
            {
                // Direct child pages are child pages context with depth 1.
                filterContext.setContext(Context.CHILD_PAGES);
                filterContext.setDepth(1);
            }
            else
            {
                // Else, parse the context.
                filterContext.setContext(Context.valueOf(searchContext));
            }
            
            if (searchContextMap.get("page") != null)
            {
                filterContext.setPageId((String) searchContextMap.get("page"));
            }
            
            @SuppressWarnings("unchecked")
            List<String> sites = Optional.of("sites")
                                         .map(sitesData::get)
                                         .map(s -> (List<String>) s)
                                         .orElse(Collections.EMPTY_LIST);
            for (String site : sites)
            {
                filterContext.addSite(site);
            }

            String contextLang = (String) entry.get("context-lang");
            if (StringUtils.isNotEmpty(contextLang))
            {
                filterContext.setContextLanguage(ContentFilter.ContextLanguage.valueOf(contextLang));
            }

            String[] tags = Optional.of("tags")
                                    .map(entry::get)
                                    .map(t -> (String[]) t)
                                    .orElse(new String[0]);
            for (String tag : tags)
            {
                filterContext.addTag(tag);
            }
            
            
            ((EventFilterSearchContext) filterContext).setOrTags(orTags);
            
            if (entry.containsKey("strict-search-on-tags"))
            {
                boolean strictSearchOnTags = (boolean) entry.get("strict-search-on-tags");
                filterContext.setTagsAutoPosting(!strictSearchOnTags);
            }
        }
    }
    
    /**
     * Date range between two dates
     * @param fromDate the start date
     * @param untilDate the end date
     */
    public record DateTimeRange(ZonedDateTime fromDate, ZonedDateTime untilDate) { /* empty */ }
}
