/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.icsreader;

import java.time.ZonedDateTime;
import java.util.Objects;

import org.ametys.plugins.calendar.events.EventsFilterHelper;

/**
 * A key class used for the events cache
 */
public class CacheKey
{
    private String _url;
    private Long _nbEvents;
    private ZonedDateTime _fromDate;
    private ZonedDateTime _untilDate;
    private Long _maxFileSize;
    
    /**
     * The value needed to create a key (so if there are many services with the same url but different parameters, we will use another cache)
     * @param url ics url
     * @param dateRange date range used to return data
     * @param nbEvents max number of events to fetch
     * @param maxFileSize max ics file size (in bytes)
     */
    public CacheKey(String url, EventsFilterHelper.DateTimeRange dateRange, Long nbEvents, Long maxFileSize)
    {
        _url = url;
        _nbEvents = nbEvents;
        if (dateRange != null)
        {
            _fromDate = dateRange.fromDate();
            _untilDate = dateRange.untilDate();
        }
        _maxFileSize = maxFileSize;
    }

    @Override
    public int hashCode()
    {
        return Objects.hash(_untilDate, _maxFileSize, _nbEvents, _fromDate, _url);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (this == obj)
        {
            return true;
        }
        if (obj == null)
        {
            return false;
        }
        if (getClass() != obj.getClass())
        {
            return false;
        }
        CacheKey other = (CacheKey) obj;
        return Objects.equals(_untilDate, other._untilDate) && Objects.equals(_maxFileSize, other._maxFileSize) && Objects.equals(_nbEvents, other._nbEvents)
                && Objects.equals(_fromDate, other._fromDate) && Objects.equals(_url, other._url);
    }
    
}
