/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.search;

import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.search.SearchResults;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.calendar.events.EventsFilterHelper.DateTimeRange;
import org.ametys.plugins.calendar.icsreader.IcsEventHelper;
import org.ametys.plugins.calendar.icsreader.IcsReader.IcsEvents;
import org.ametys.plugins.calendar.icsreader.LocalVEvent;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstance;
import org.ametys.web.frontoffice.search.requesttime.SearchComponent;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.frontoffice.search.requesttime.impl.SaxResultsSearchComponent;
import org.ametys.web.repository.page.ZoneItem;

/**
 * {@link SearchComponent} to sax calendar events results resulting of contents' search and/or external ICS
 *
 */
public class CalendarSearchComponent extends SaxResultsSearchComponent implements Serviceable
{
    private IcsEventHelper _icsEventHelper;
    private AmetysObjectResolver _resolver;

    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _icsEventHelper = (IcsEventHelper) smanager.lookup(IcsEventHelper.ROLE);
    }

    @Override
    public boolean supports(SearchComponentArguments args)
    {
        return CalendarSearchService.isFormSubmit(args) && CalendarSearchService.isActive(args);
    }

    @Override
    public void execute(SearchComponentArguments args) throws Exception
    {
        SearchResults<AmetysObject> aoResults = getResults(args);

        String zoneItemId = args.serviceInstance().getId();
        ZoneItem zoneItem = (ZoneItem) _resolver.resolveById(zoneItemId);

        DateTimeRange dateRange = CalendarSearchService.getDateRange(args);
        List<IcsEvents> icsResults;

        if (args.generatorParameters().getParameterAsBoolean("export", false))
        {
            // If we are exporting the ICS, we do not want to use maximum events or ICS file size limitation
            icsResults = _icsEventHelper.getICSEvents(zoneItem, args.currentSite().getName(), dateRange, Long.MAX_VALUE, Long.MAX_VALUE);
        }
        else
        {
            icsResults = _icsEventHelper.getICSEvents(zoneItem, args.currentSite().getName(), dateRange);
        }

        ContentHandler handler = args.contentHandler();

        int aoTotal = total(aoResults, args.serviceInstance());
        int total = aoTotal + icsResults.size();

        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("total", String.valueOf(total));
        XMLUtils.startElement(handler, "hits", atts);

        SearchServiceInstance service = args.serviceInstance();

        // Content's hits
        saxHits(aoResults, 0, args, service.getReturnables(), service.getAdditionalParameterValues());

        // ICS hits
        List<String> selectedTags = CalendarSearchService.getSelectedTags(args);
        saxICSHits(handler, icsResults, dateRange, selectedTags, aoTotal);
        
        XMLUtils.endElement(handler, "hits");

        saxDangeRange(handler, dateRange, total);

        _icsEventHelper.saxICSErrors(icsResults, handler);

    }

    /**
     * SAX results resulting of external ICS
     * @param handler the content handler
     * @param icsResults the ICS events
     * @param dateRange the date range
     * @param filteredTags A list of tag's name to filter ICS events. If empty no filter is applied.
     * @param start the start index
     * @throws SAXException if an error occured
     */
    protected void saxICSHits(ContentHandler handler, List<IcsEvents> icsResults, DateTimeRange dateRange, List<String> filteredTags, int start) throws SAXException
    {
        Pair<List<LocalVEvent>, String> localIcsEvents = _icsEventHelper.toLocalIcsEvent(icsResults, dateRange, filteredTags);

        List<LocalVEvent> icsHits = localIcsEvents.getLeft();
        _icsEventHelper.saxIcsEventHits(handler, icsHits, start);

        String fullICSDistantEvents = localIcsEvents.getRight();
        XMLUtils.createElement(handler, "rawICS", fullICSDistantEvents);
        
        // Generate VTimeZones for distant events
        String timezones = _icsEventHelper.toVTimeZone(icsResults, dateRange, List.of());

        XMLUtils.createElement(handler, "timezones", timezones);
    }

    /**
     * SAx date range
     * @param handler the content handler
     * @param dateRange the date range
     * @param total the total hits in date range
     * @throws SAXException if an error occured
     */
    protected void saxDangeRange(ContentHandler handler, DateTimeRange dateRange, int total) throws SAXException
    {
        if (dateRange != null)
        {
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("total", String.valueOf(total));
            atts.addCDATAAttribute("start", DateUtils.zonedDateTimeToString(dateRange.fromDate()));
            atts.addCDATAAttribute("end", DateUtils.zonedDateTimeToString(dateRange.untilDate()));

            XMLUtils.createElement(handler, "date-range", atts);
        }
    }
}
