/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.calendar.search;

import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.tag.ColorableTag;
import org.ametys.cms.tag.Tag;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.calendar.events.EventsFilterHelper.DateTimeRange;
import org.ametys.web.frontoffice.search.SearchService;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstance;
import org.ametys.web.frontoffice.search.instance.model.RightCheckingMode;
import org.ametys.web.frontoffice.search.metamodel.AdditionalParameterValueMap;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Calendar search service
 *
 */
public class CalendarSearchService extends SearchService
{
    /** Id of service */
    public static final String SERVICE_ID = "org.ametys.plugins.calendar.SearchEvents";
    
    /** Parameter to enable calendar search components */
    public static final String ENABLE_CALENDAR_PARAMETER_NAME = "calendar";
    
    /** Service parameter for tags' categories */
    public static final String SERVICE_PARAM_TAG_CATEGORIES = "tag-categories";
    
    /** Service parameter for contents' view */
    public static final String SERVICE_PARAM_CONTENT_VIEW = "calendarContentView";
    
    /** Service parameter for calendar content types */
    public static final String SERVICE_PARAM_CONTENT_TYPES = "calendarContentTypes";
    
    /** The parameter name for saving user preferences */
    public static final String SERVICE_PARAM_SAVE_USER_PREFS = "saveFilterInUserPrefs";
    
    /** Parameter for start date */
    public static final String PARAM_START_DATE = "start";
    
    /** Parameter for end date */
    public static final String PARAM_END_DATE = "end";
    
    /** Parameter for selected tags */
    public static final String PARAM_TAGS = "tags";
    
    /** Parameter for selected tags */
    public static final String PARAM_SUBMIT = "submit-form";
    
    /** Attribute name for content's start date */
    public static final String START_DATE_ATTRIBUTE_NAME = "start-date";
    
    /** Attribute name for content's end date */
    public static final String END_DATE_ATTRIBUTE_NAME = "end-date";
    
    private static final String __TAGS_ALL_VALUE = "_ALL";
    
    private static final String __ENABLE_SAVE_USER_PREFS_PARAMETER_NAME = "saveUserPrefs";
    
    @Override
    public boolean isCacheable(Page currentPage, ZoneItem zoneItem)
    {
        boolean isDebug = _isDebug(ContextHelper.getRequest(_context), _renderingContextHandler);
        SearchServiceInstance serviceInstance = _searchServiceInstanceManager.get(zoneItem.getId());
        
        if (!isDebug)
        {
            // Cacheable if no user input and right checking mode is on 'FAST'
            return !_hasUserInput(serviceInstance) && serviceInstance.getRightCheckingMode() != RightCheckingMode.EXACT && !hasTagCategories(serviceInstance);
        }
        
        return false;
    }
    
    /**
     * Determines if tags' categories filter is active
     * @param serviceInstance the service instance
     * @return true if tags' categories filter is active
     */
    public static boolean hasTagCategories(SearchServiceInstance serviceInstance)
    {
        List<String> categoryNames = getTagCategories(serviceInstance);
        return categoryNames != null && !categoryNames.isEmpty();
    }
    
    /**
     * Get the tags' category
     * @param serviceInstance the service instance
     * @return the tags' category. Can be null if no tags' categories are configured
     */
    public static List<String> getTagCategories(SearchServiceInstance serviceInstance)
    {
        AdditionalParameterValueMap additionalParameterValues = serviceInstance.getAdditionalParameterValues();
        return additionalParameterValues.getValue(SERVICE_PARAM_TAG_CATEGORIES);
    }
    
    /**
     * Returns <code>true</code> if save user preferences is enabled
     * @param args the search arguments
     * @return <code>true</code> if save user preferences is enabled
     */
    public static boolean saveUserPrefsEnabled(SearchComponentArguments args)
    {
        return args.generatorParameters().getParameterAsBoolean(__ENABLE_SAVE_USER_PREFS_PARAMETER_NAME, true);
    }
    
    /**
     * Returns <code>true</code> if selected tags filter has to be saved in user preferences
     * @param serviceInstance the service instance
     * @return <code>true</code> if selected tags filtes has to be saved in user preferences
     */
    public static boolean saveFilterInUserPrefs(SearchServiceInstance serviceInstance)
    {
        AdditionalParameterValueMap additionalParameterValues = serviceInstance.getAdditionalParameterValues();
        return additionalParameterValues.getValue(SERVICE_PARAM_SAVE_USER_PREFS);
    }
    
    /**
     * Get the active tags
     * @param args the search arguments
     * @return the active tags
     */
    public static List<String> getSelectedTags(SearchComponentArguments args)
    {
        String parameter = args.generatorParameters().getParameter(PARAM_TAGS, "");
        if (__TAGS_ALL_VALUE.equals(parameter))
        {
            return List.of();
        }
        String[] selectedTags = StringUtils.isNotEmpty(parameter) ? parameter.split(",") : new String[0];
        return Arrays.asList(selectedTags);
    }
    
    /**
     * Determines if is calendar service
     * @param args the search component arguments
     * @return true if it is calendar service
     */
    public static boolean isActive(SearchComponentArguments args)
    {
        return args.generatorParameters().getParameterAsBoolean(ENABLE_CALENDAR_PARAMETER_NAME, false);
    }
    
    /**
     * Determines if form is submit to get results
     * @param args the search component arguments
     * @return true if form is submit
     */
    public static boolean isFormSubmit(SearchComponentArguments args)
    {
        return args.generatorParameters().getParameterAsBoolean(PARAM_SUBMIT, false);
    }
    
    /**
     * Get search date range
     * @param args the search component arguments
     * @return the date time range
     */
    public static DateTimeRange getDateRange(SearchComponentArguments args)
    {
        Parameters parameters = args.generatorParameters();
        if (parameters.isParameter(PARAM_START_DATE) && parameters.isParameter(PARAM_END_DATE))
        {
            try
            {
                String start = URIUtils.decode(parameters.getParameter(CalendarSearchService.PARAM_START_DATE));
                String end = URIUtils.decode(parameters.getParameter(CalendarSearchService.PARAM_END_DATE));
                
                ZonedDateTime fromDateTime = DateUtils.parseZonedDateTime(start);
                ZonedDateTime untilDateTime = DateUtils.parseZonedDateTime(end);
                
                return new DateTimeRange(fromDateTime, untilDateTime);
            }
            catch (ParameterException e)
            {
                // ignore
            }
        }
        
        return null;
    }
    
    /**
     * SAX a tag
     * @param handler the content handler
     * @param tag the tag
     * @throws SAXException if an error occurred
     */
    public static void saxTag(ContentHandler handler, Tag tag) throws SAXException
    {
        saxTag(handler, new AttributesImpl(), tag);
    }
    
    /**
     * SAX a tag
     * @param handler the content handler
     * @param attrs The attributes 
     * @param tag the tag
     * @throws SAXException if an error occurred
     */
    public static void saxTag(ContentHandler handler, AttributesImpl attrs, Tag tag) throws SAXException
    {
        if (tag != null)
        {
            attrs.addCDATAAttribute("name", tag.getName());
            XMLUtils.startElement(handler, "tag", attrs);
            
            XMLUtils.startElement(handler, "label");
            tag.getTitle().toSAX(handler);
            XMLUtils.endElement(handler, "label");
            
            saxTagColor(handler, tag);
            
            XMLUtils.endElement(handler, "tag");
        }
    }
    
    /**
     * SAX the color of a tag
     * @param handler the content handler
     * @param tag the tag
     * @throws SAXException if an error occurred
     */
    public static void saxTagColor(ContentHandler handler, Tag tag) throws SAXException
    {
        if (tag instanceof ColorableTag)
        {
            String colorIndex = ((ColorableTag) tag).getColor(true);
            Map<String, String> colors = ((ColorableTag) tag).getColorComponent().getColors().get(colorIndex);
            if (colors != null)
            {
                XMLUtils.startElement(handler, "color");
                for (Entry<String, String> entry : colors.entrySet())
                {
                    XMLUtils.createElement(handler, entry.getKey(), entry.getValue());
                }
                XMLUtils.endElement(handler, "color");
            }
        }
    }
}
