/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.cart;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.cms.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.cms.data.holder.ModifiableIndexableDataHolder;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.cms.indexing.solr.SolrAclCacheUninfluentialObject;
import org.ametys.cms.repository.Content;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultAmetysObject;

/**
 * Class representing a cart, backed by a JCR node.<br>
 */
@SolrAclCacheUninfluentialObject
public class Cart extends DefaultAmetysObject<CartFactory> implements ModifiableModelAwareDataAwareAmetysObject
{
    /** Property name for cart title */
    public static final String TITLE = "label";
    /** Property name for cart description */
    public static final String DESCRIPTION = "description";
    /** Property name for cart documentation */
    public static final String DOCUMENTATION = "documentation";
    /** Property name for cart author */
    public static final String AUTHOR = "author";
    /** Property name for cart last contributor */
    public static final String CONTRIBUTOR = "contributor";
    /** Property name for cart creation date */
    public static final String CREATIONDATE = "creationDate";
    /** Property name for cart last modification date */
    public static final String LASTMODIFICATIONDATE = "lastModificationDate";

    /** Property name for cart content elements */
    public static final String CONTENT_CART_ELEMENTS = "contents";
    /** Property name for cart resource elements */
    public static final String RESOURCE_CART_ELEMENTS = "resources";
    /** Property name for cart queries elements */
    public static final String QUERIES_CART_ELEMENTS = "queries";
    /** Property name for cart queries from directory elements */
    public static final String QUERIES_FROM_DIRECTORY_CART_ELEMENTS = "queries-from-directory";
    
    /** Property name for cart query id */
    public static final String QUERY_ID_PROPERTY = "id";
    /** Property name for cart query description */
    public static final String QUERY_DESCRIPTION_PROPERTY = "description";
    /** Property name for cart query author */
    public static final String QUERY_AUTHOR_PROPERTY = "author";
    /** Property name for cart query title */
    public static final String QUERY_TITLE_PROPERTY = "title";
    /** Property name for cart query creation date */
    public static final String QUERY_DATE_PROPERTY = "date";
    
    private static Logger _logger = LoggerFactory.getLogger(Cart.class.getName());
    
    /**
     * Rights profiles
     */
    public enum CartProfile
    {
        /** Read access */
        READ_ACCESS,
        /** Write access */
        WRITE_ACCESS,
        /** Right access */
        RIGHT_ACCESS;
        
        @Override
        public String toString()
        {
            return name().toLowerCase();
        }
    }
    
    /**
     * Types of CartElement
     */
    public enum CartElementType
    {
        /** Type: content. */
        CONTENT,
        /** Type: resource. */
        RESOURCE,
        /** Type: cartQuery. */
        CARTQUERY,
        /** Type: query from directory. */
        CARTQUERYFROMDIRECTORY
    }
    
    /**
     * Creates an {@link Cart}.
     * @param node the node backing this {@link AmetysObject}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the DefaultAmetysObjectFactory which created the AmetysObject
     */
    public Cart(Node node, String parentPath, CartFactory factory)
    {
        super(node, parentPath, factory);
    }
    
    /**
     * Set the title of this cart.
     * @param title the title
     */
    public void setTitle(String title)
    {
        this.setValue(TITLE, title);
    }
    
    /**
     * Set the description of this cart.
     * @param description the description
     */
    public void setDescription(String description)
    {
        this.setValue(DESCRIPTION, description);
    }
    
    /**
     * Set the documentation of this cart.
     * @param documentation the documentation
     */
    public void setDocumentation(String documentation)
    {
        this.setValue(DOCUMENTATION, documentation);
    }
    
    /**
     * Set the author of this cart.
     * @param author the author
     */
    public void setAuthor(UserIdentity author)
    {
        this.setValue(AUTHOR, author);
    }

    /**
     * Set the last contributor of this cart.
     * @param contributor the last contributor
     */
    public void setContributor(UserIdentity contributor)
    {
        this.setValue(CONTRIBUTOR, contributor);
    }
    
    /**
     * Set the date of the creation.
     * @param creationDate the last modification date to set.
     */
    public void setCreationDate(ZonedDateTime creationDate)
    {
        this.setValue(CREATIONDATE, creationDate);
    }
    
    /**
     * Set the date of the last modification.
     * @param lastModificationDate the last modification date to set.
     */
    public void setLastModificationDate(ZonedDateTime lastModificationDate)
    {
        this.setValue(LASTMODIFICATIONDATE, lastModificationDate);
    }
    
    /**
     * Get the title of the cart
     * @return The title
     */
    public String getTitle()
    {
        return this.getValue(TITLE);
    }
        
    /**
     * Get the description of the cart
     * @return The description
     */
    public String getDescription()
    {
        return this.getValue(DESCRIPTION);
    }

    /**
     * Get the documentation of the cart
     * @return The documentation
     */
    public String getDocumentation()
    {
        return this.getValue(DOCUMENTATION);
    }
    
    /**
     * Get the author of the cart
     * @return The author
     */
    public UserIdentity getAuthor()
    {
        return this.getValue(AUTHOR);
    }
    
    /**
     * Get the last contributor of the cart
     * @return The contributor
     */
    public UserIdentity getContributor()
    {
        return this.getValue(CONTRIBUTOR);
    }
    
    /**
     * Get the date of the last modification of the cart
     * @return The date
     */
    public ZonedDateTime getCreationDate()
    {
        return this.getValue(CREATIONDATE);
    }
    
    /**
     * Get the date of the last modification of the cart
     * @return The date
     */
    public ZonedDateTime getLastModificationDate()
    {
        return this.getValue(LASTMODIFICATIONDATE);
    }
        
    /**
     * Add a content to the cart
     * @param contentId The content id
     */
    public void addContent (String contentId)
    {
        _addElementToCart(contentId, CONTENT_CART_ELEMENTS);
    }
    
    /**
     * Add a resource to the cart
     * @param resourceId The resource id
     */
    public void addResource (String resourceId)
    {
        _addElementToCart(resourceId, RESOURCE_CART_ELEMENTS);
    }
    
    /**
     * Add a query from directory to the cart
     * @param queryId The query id
     */
    public void addQueryFormDirectory (String queryId)
    {
        _addElementToCart(queryId, QUERIES_FROM_DIRECTORY_CART_ELEMENTS);
    }

    private void _addElementToCart(String elementId, String attributePath)
    {
        String[] elmtArray = getValue(attributePath, false, new String[0]);
        Set<String> elmts = new HashSet<>();
        Collections.addAll(elmts, elmtArray);
        elmts.add(elementId);
        setValue(attributePath, elmts.toArray(new String[elmts.size()]));
    }
    
    /**
     * Add a query to the cart
     * @param author The author of the query
     * @param title The title of the query
     * @param description The query as string
     */
    public void addQuery (UserIdentity author, String title, String description)
    {
        addQuery(org.ametys.core.util.StringUtils.generateKey(), author, title, description, ZonedDateTime.now());
    }
    
    /**
     * Add a query to the cart
     * @param id The id of the query
     * @param author The author of the query
     * @param title The title of the query
     * @param description The query as string
     * @param date The creation date of the query
     */
    public void addQuery (String id, UserIdentity author, String title, String description, ZonedDateTime date)
    {
        ModifiableModelAwareRepeater queries = getRepeater(QUERIES_CART_ELEMENTS, true);
        ModifiableModelAwareRepeaterEntry query = queries.addEntry();
        query.setValue(QUERY_ID_PROPERTY, id);
        query.setValue(QUERY_TITLE_PROPERTY, title);
        query.setValue(QUERY_DESCRIPTION_PROPERTY, description);
        query.setValue(QUERY_AUTHOR_PROPERTY, author);
        query.setValue(QUERY_DATE_PROPERTY, date);
    }
    
    /**
     * Delete an element
     * @param elmtId The id of element to remove
     * @param elmtType The type of element to remove
     */
    public void removeElement (String elmtId, CartElementType elmtType)
    {
        switch (elmtType)
        {
            case CONTENT:
                _removeContent(elmtId);
                break;
                
            case RESOURCE:
                _removeResource(elmtId);
                break;
                
            case CARTQUERY:
                _removeQuery(elmtId);
                break;
                
            case CARTQUERYFROMDIRECTORY:
                _removeQueryFromDirectory(elmtId);
                break;
            
            default:
                break;
        }
    }
    
    /**
     * Remove a resource from the cart
     * @param resourceIdToRemove The id of the resource to remove
     */
    protected void _removeResource(String resourceIdToRemove)
    {
        _removeElementFromCart(resourceIdToRemove, RESOURCE_CART_ELEMENTS);
    }
    
    /**
     * Remove a content from the cart
     * @param contentIdToRemove The id of the content to remove
     */
    protected void _removeContent(String contentIdToRemove)
    {
        _removeElementFromCart(contentIdToRemove, CONTENT_CART_ELEMENTS);
    }
    
    /**
     * Remove a query from directory from the cart
     * @param queryIdToRemove The id of the query from directory to remove
     */
    protected void _removeQueryFromDirectory(String queryIdToRemove)
    {
        _removeElementFromCart(queryIdToRemove, QUERIES_FROM_DIRECTORY_CART_ELEMENTS);
    }
    
    private void _removeElementFromCart(String elementId, String attributePath)
    {
        String[] elements = getValue(attributePath, false, new String[0]);
        elements = ArrayUtils.removeElement(elements, elementId);
        setValue(attributePath, elements);
    }
    
    /**
     * Remove a query from the cart
     * @param queryIdToRemove The id of the query to remove
     */
    protected void _removeQuery(String queryIdToRemove)
    {
        ModifiableModelAwareRepeater queries = getRepeater(QUERIES_CART_ELEMENTS, true);
        ModifiableModelAwareRepeaterEntry entryToRemove = queries.getEntries()
                .stream()
              .filter(entry -> entry.getValue(QUERY_ID_PROPERTY).equals(queryIdToRemove))
              .findFirst()
              .orElse(null);
        
        if (entryToRemove != null)
        {
            queries.removeEntry(entryToRemove.getPosition());
        }
    }
    
    /**
     * Get the elements of the cart
     * @return The elements of the cart
     */
    public List<CartElement> getElements ()
    {
        List<CartElement> elmts = new ArrayList<>();
        
        try
        {
            elmts.addAll(getContentCartElements());
            elmts.addAll(getResourceCartElements());
            elmts.addAll(getQueryCartElements());
            elmts.addAll(getQueryFromDirectoryCartElements());
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error while getting cart elements", e);
        }
        
        return elmts;
    }
    
    /**
     * Get the contents of the cart
     * @return The elements of the cart
     * @throws RepositoryException if an exception occurs while exploring the repository
     */
    public List<ContentElement> getContentCartElements () throws RepositoryException
    {
        List<String> unexistingElmts = new ArrayList<>();
        
        List<ContentElement> elmts = new ArrayList<>();
        
        String[] contents = getValue(CONTENT_CART_ELEMENTS, false, new String[0]);
        
        for (String contentId : contents)
        {
            try
            {
                Content content = _getFactory().getResolver().resolveById(contentId);
                elmts.add(new ContentElement(content, _getFactory()._getContentTypesHelper(), _getFactory()._getContentTypeEP()));
            }
            catch (UnknownAmetysObjectException e)
            {
                _logger.error("The content of id '{}' does not exist anymore. It will be deleting from cart '{}'.", contentId, getId());
                unexistingElmts.add(contentId);
            }
        }
        // Delete unexisting contents
        for (String contentId : unexistingElmts)
        {
            _removeContent(contentId);
        }
        
        return elmts;
    }
    
    /**
     * Get the resources of the cart
     * @return The elements of the cart
     * @throws RepositoryException if an exception occurs while exploring the repository
     */
    public List<ResourceElement> getResourceCartElements () throws RepositoryException
    {
        List<String> unexistingElmts = new ArrayList<>();
        List<ResourceElement> elmts = new ArrayList<>();

        
        String[] resources = getValue(RESOURCE_CART_ELEMENTS, false, new String[0]);
        
        for (String resourceId : resources)
        {
            try
            {
                Resource resource = _getFactory().getResolver().resolveById(resourceId);
                elmts.add(new ResourceElement(resource));
            }
            catch (UnknownAmetysObjectException e)
            {
                _logger.error("The resource of id '{}' does not exist anymore. It will be deleting from cart '{}'.", resourceId, getId());
                unexistingElmts.add(resourceId);
            }
        }
        
        // Delete unexisting resources
        for (String resourceId : unexistingElmts)
        {
            _removeResource(resourceId);
        }
        
        return elmts;
    }
    
    /**
     * Get the resources of the cart
     * @return The elements of the cart
     * @throws RepositoryException if an exception occurs while exploring the repository
     */
    public List<QueryFromDirectoryElement> getQueryFromDirectoryCartElements () throws RepositoryException
    {
        List<String> unexistingElmts = new ArrayList<>();
        List<QueryFromDirectoryElement> elmts = new ArrayList<>();

        
        String[] queries = getValue(QUERIES_FROM_DIRECTORY_CART_ELEMENTS, false, new String[0]);
        for (String queryId : queries)
        {
            try
            {
                Query query = _getFactory().getResolver().resolveById(queryId);
                elmts.add(new QueryFromDirectoryElement(query));
            }
            catch (UnknownAmetysObjectException e)
            {
                _logger.error("The query of id '{}' does not exist anymore. It will be deleting from cart '{}'.", queryId, getId());
                unexistingElmts.add(queryId);
            }
        }
        
        // Delete unexisting queries
        for (String queryId : unexistingElmts)
        {
            _removeQueryFromDirectory(queryId);
        }
        
        return elmts;
    }
    
    /**
     * <code>true</code> if the query cart element exist
     * @param queryId the query id
     * @return <code>true</code> if the query cart element exist
     */
    public boolean hasQueryFromDirectoryCartElement(String queryId)
    {
        String[] queries = getValue(QUERIES_FROM_DIRECTORY_CART_ELEMENTS, false, new String[0]);
        return ArrayUtils.contains(queries, queryId);
    }
    
    /**
     * Get the queries of the cart
     * @return The elements of the cart
     * @throws RepositoryException if an exception occurs while exploring the repository
     */
    public List<QueryElement> getQueryCartElements () throws RepositoryException
    {
        ModifiableModelAwareRepeater queries = getRepeater(QUERIES_CART_ELEMENTS, true);
        
        List<QueryElement> elmts = queries.getEntries()
                .stream()
                .map(entry -> {
                    String id = entry.getValue(QUERY_ID_PROPERTY);
                    UserIdentity author = entry.getValue(QUERY_AUTHOR_PROPERTY);
                    String query = entry.getValue(QUERY_DESCRIPTION_PROPERTY);
                    String title = entry.getValue(QUERY_TITLE_PROPERTY);
                    ZonedDateTime creationDate = entry.getValue(QUERY_DATE_PROPERTY);
                           
                    return new QueryElement(id, query, author, creationDate, author, creationDate, title);
                })
                .collect(Collectors.toList());
        
        return elmts;
    }

    public ModifiableIndexableDataHolder getDataHolder()
    {
        JCRRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelAwareDataHolder(repositoryData, this._getFactory().getModel());
    }
}
