/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.cart;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.slf4j.LoggerFactory;

import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.cart.Cart.CartElementType;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.NameHelper;

/**
 * DAO for manipulating carts
 */
public class CartsDAO implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = CartsDAO.class.getName();
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The current user provider */
    private CurrentUserProvider _userProvider;
    
    /** The Ametys object resolver */
    private AmetysObjectResolver _resolver;

    private UserHelper _userHelper;

    private RightManager _rightManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _userProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    /**
     * Gets carts information.
     * @param cartIds The ids of the carts to retrieve.
     * @return The carts information
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> getCartsInformation (List<String> cartIds)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<Map<String, Object>> carts = new LinkedList<>();
        List<Map<String, Object>> notAllowedCarts = new LinkedList<>();
        Set<String> unknownCarts = new HashSet<>();
        
        for (String id : cartIds)
        {
            Cart cart = getCart(id);
            
            if (cart != null)
            {
                if (_hasRight(cart))
                {
                    carts.add(getCartProperties(cart));
                }
                else
                {
                    notAllowedCarts.add(getCartProperties(cart));
                }
                
            }
            else
            {
                unknownCarts.add(id);
            }
        }
        
        result.put("carts", carts);
        result.put("unknownCarts", unknownCarts);
        result.put("notAllowedCarts", notAllowedCarts);
        
        return result;
    }
    
    /**
     * Creates a cart.
     * @param title The title
     * @param description The description
     * @param documentation The documentation of the cart
     * @return The id of the created cart
     */
    @Callable(rights = "Plugin_Cart_Rights_Create")
    public Map<String, String> createCart (String title, String description, String documentation)
    {
        Map<String, String> result = new HashMap<>();
        
        ModifiableTraversableAmetysObject cartsNode = CartHelper.getCartsNode(_resolver);
        
        String name = NameHelper.filterName(title);
        
        // Find unique name
        int index = 1;
        String originalName = name;
        while (cartsNode.hasChild(name))
        {
            name = originalName + "_" + (++index);
        }
        
        // title might be modified to handle project with same title.
        String realTitle = title + (index > 1 ? " (" + index + ")" : "");
        
        Cart cart = cartsNode.createChild(name, CartFactory.CART_NODETYPE);
        cart.setTitle(realTitle);
        cart.setDescription(description);
        cart.setDocumentation(documentation);
        cart.setAuthor(_userProvider.getUser());
        cart.setContributor(_userProvider.getUser());
        cart.setCreationDate(ZonedDateTime.now());
        cart.setLastModificationDate(ZonedDateTime.now());
        
        cartsNode.saveChanges();
        
        result.put("id", cart.getId());
        
        return result;
    }
    
    /**
     * Updates a cart.
     * @param id The id of the cart to update
     * @param title The title
     * @param description The description
     * @param documentation The documentation of the cart
     * @return The id of the updated cart
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, String> updateCart (String id, String title, String description, String documentation)
    {
        Map<String, String> result = new HashMap<>();
        
        Cart cart = _resolver.resolveById(id);
        
        if (canWrite(_userProvider.getUser(), cart))
        {
            cart.setTitle(title);
            cart.setDescription(description);
            cart.setDocumentation(documentation);
            cart.setContributor(_userProvider.getUser());
            cart.setLastModificationDate(ZonedDateTime.now());
            cart.saveChanges();
        }
        else
        {
            result.put("message", "not-allowed");
        }
        
        result.put("id", cart.getId());
        
        return result;
    }
    
    /**
     * Deletes some carts.
     * @param ids The ids of the carts to delete
     * @return The ids of the deleted carts
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteCarts (List<String> ids)
    {
        Map<String, Object> result = new HashMap<>();
        
        List<String> deletedCarts = new ArrayList<>();
        List<String> unknownCarts = new ArrayList<>();
        List<String> notallowedCarts = new ArrayList<>();
        
        for (String id : ids)
        {
            try
            {
                Cart cart = _resolver.resolveById(id);
                if (canWrite(_userProvider.getUser(), cart))
                {
                    cart.remove();
                    cart.saveChanges();
                    deletedCarts.add(id);
                }
                else
                {
                    notallowedCarts.add(cart.getTitle());
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                unknownCarts.add(id);
                LoggerFactory.getLogger(getClass()).error("Unable to delete cart. The cart of id '" + id + " doesn't exist", e);
            }
        }
        
        result.put("deletedCarts", deletedCarts);
        result.put("notallowedCarts", notallowedCarts);
        result.put("unknownCarts", unknownCarts);
        
        return result;
    }
    
    /**
     * Add elements to a cart.
     * @param cartId The cart id.
     * @param type The type of element.
     * @param elementParams The parameters of the element.
     * @return The id of the cart or an error
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> addElements (String cartId, String type, Map<String, Object> elementParams)
    {
        Map<String, Object> result = new HashMap<>();
        
        UserIdentity user = _userProvider.getUser();
        
        Cart cart = _resolver.resolveById(cartId);
        
        if (!canWrite(user, cart))
        {
            LoggerFactory.getLogger(getClass()).error("User '{}' try to add elements to a cart without convenient privileges", user);
            result.put("msg", "not-allowed");
            return result;
        }
        
        switch (CartElementType.valueOf(type.toUpperCase()))
        {
            case CONTENT:
                @SuppressWarnings("unchecked")
                List<String> contentIds = (List<String>) elementParams.get("ids");
                for (String contentId : contentIds)
                {
                    cart.addContent(contentId);
                }
                break;

            case RESOURCE:
                @SuppressWarnings("unchecked")
                List<String> resourceIds = (List<String>) elementParams.get("ids");
                for (String resourceId : resourceIds)
                {
                    cart.addResource(resourceId);
                }
                break;
                
            case CARTQUERY:
                String title = (String) elementParams.get("title");
                String description = (String) elementParams.get("description");
                
                cart.addQuery(user, title, description);
                break;
            
            case CARTQUERYFROMDIRECTORY:
                @SuppressWarnings("unchecked")
                List<String> queryIds = (List<String>) elementParams.get("queryIds");
                for (String queryId : queryIds)
                {
                    cart.addQueryFormDirectory(queryId);
                }
                break;
                
            default:
                throw new IllegalArgumentException("Unknown cart element type");
        }

        cart.setContributor(_userProvider.getUser());
        cart.setLastModificationDate(ZonedDateTime.now());
        cart.saveChanges();
        
        result.put("id", cart.getId());
        
        return result;
    }
    
    /**
     * Deletes elements of a cart.
     * @param cartId The cart id.
     * @param cartElements The elements to delete.
     * @return The id of the cart or an error
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> deleteElements (String cartId, List<Map<String, String>> cartElements)
    {
        Map<String, Object> result = new HashMap<>();
        
        Cart cart = _resolver.resolveById(cartId);
        
        UserIdentity user = _userProvider.getUser();
        
        if (!canWrite(user, cart))
        {
            LoggerFactory.getLogger(getClass()).error("User '{}' try to add elements to a cart without convenient privileges", user);
            result.put("msg", "not-allowed");
            return result;
        }
        
        for (Map<String, String> cartElement : cartElements)
        {
            CartElementType type = CartElementType.valueOf(cartElement.get("type").toUpperCase());
            cart.removeElement(cartElement.get("id"), type);
        }

        cart.setContributor(_userProvider.getUser());
        cart.setLastModificationDate(ZonedDateTime.now());
        cart.saveChanges();
       
        result.put("id", cartId);
        
        return result;
    }
    
    /**
     * Get the cart with given id
     * @param cartId The cart id
     * @return The retrieved cart or null.
     */
    public Cart getCart(String cartId)
    {
        try
        {
            return _resolver.resolveById(cartId);
        }
        catch (AmetysRepositoryException e)
        {
            if (LoggerFactory.getLogger(getClass()).isWarnEnabled())
            {
                LoggerFactory.getLogger(getClass()).warn("Failed to retrieves the cart with id : " +  cartId, e);
            }
            
            return null;
        }
    }
    
    /**
     * Get the cart type properties
     * @param cart The cart
     * @return The cart type properties
     */
    public Map<String, Object> getCartProperties (Cart cart)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", cart.getId());
        infos.put("title", cart.getTitle());
        infos.put("description", cart.getDescription());
        infos.put("documentation", cart.getDocumentation());
        infos.put("author", _userHelper.user2json(cart.getAuthor()));
        infos.put("contributor", _userHelper.user2json(cart.getContributor())); 
        infos.put("lastModificationDate", DateUtils.zonedDateTimeToString(cart.getLastModificationDate()));
        infos.put("creationDate", DateUtils.zonedDateTimeToString(cart.getCreationDate()));
        
        UserIdentity currentUser = _userProvider.getUser();
        infos.put("canRead", canRead(currentUser, cart));
        infos.put("canWrite", canWrite(currentUser, cart));
        infos.put("canAssignRights", canAssignRights(currentUser, cart));
        
        return infos;
    }
    
    /**
     * Test if the current user has the right needed by the content type to view this cart.
     * @param cart The cart
     * @return true if the user has the right needed, false otherwise.
     */
    protected boolean _hasRight(Cart cart)
    {
        UserIdentity user = _userProvider.getUser();
        return canRead(user, cart) || canWrite(user, cart);
    }
    
    /**
     * Check if a user have read rights on a cart
     * @param userIdentity the user
     * @param cart the cart
     * @return true if the user have read rights on a cart
     */
    public boolean canRead(UserIdentity userIdentity, Cart cart)
    {
        return _rightManager.hasReadAccess(userIdentity, cart);
    }

    /**
     * Check if a user have write rights on a cart
     * @param userIdentity the user
     * @param cart the cart
     * @return true if the user have write rights on a cart
     */
    public boolean canWrite(UserIdentity userIdentity, Cart cart)
    {
        return _rightManager.hasRight(userIdentity, "Cart_Rights_Admin", cart) == RightResult.RIGHT_ALLOW;
    }

    /**
     * Check if a user have rights to edit rights on a cart
     * @param userIdentity the user
     * @param cart the cart
     * @return true if the user have write rights to edit rights on a cart
     */
    public boolean canAssignRights(UserIdentity userIdentity, Cart cart)
    {
        return canWrite(userIdentity, cart) || _rightManager.hasRight(userIdentity, "Runtime_Rights_Rights_Handle", "/cms") == RightResult.RIGHT_ALLOW;
    }

}
