/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.cart.accesscontroller;

import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.plugins.cart.Cart;
import org.ametys.plugins.cart.CartContainer;
import org.ametys.plugins.cart.CartHelper;
import org.ametys.plugins.core.impl.right.AbstractHierarchicalAccessController;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * {@link AccessController} for a {@link Cart}
 */
public class CartAccessController extends AbstractHierarchicalAccessController<AmetysObject>
{
    /** cart context category */
    public static final I18nizableText CART_CONTEXT_CATEGORY = new I18nizableText("plugin.cart", "PLUGINS_CART_RIGHTS_CATEGORY");
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public boolean supports(Object object)
    {
        return object instanceof Cart || object instanceof CartContainer;
    }
    
    @Override
    protected Set<AmetysObject> _getParents(AmetysObject object)
    {
        if (object instanceof Cart)
        {
            return Set.of(object.getParent());
        }
        else
        {
            return null;
        }
    }
    
    @Override
    protected boolean ignoreOnHasAnyPermission()
    {
        return true;
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        if (workspacesContexts.contains("/cms"))
        {
            return Set.of(CartHelper.getCartsNode(_resolver));
        }
        return null;
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        if (object instanceof Cart)
        {
            Map<String, I18nizableTextParameter> params = Map.of("title", getObjectLabel(object));
            return new I18nizableText("plugin.cart", "PLUGINS_CART_ACCESS_CONTROLLER_CONTEXT_EXPLANATION_LABEL", params);
        }
        else if (object instanceof CartContainer)
        {
            return new I18nizableText("plugin.cart", "PLUGINS_CART_ACCESS_CONTROLLER_ROOT_CONTEXT_EXPLANATION_LABEL");
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Cart cart)
        {
            return new I18nizableText(cart.getTitle());
        }
        else if (object instanceof CartContainer)
        {
            return new I18nizableText("plugin.cart", "PLUGINS_CART_ACCESS_CONTROLLER_ROOT_CONTEXT_LABEL");
        }
        throw new RightsException("Unsupported object " + object.toString());
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return CART_CONTEXT_CATEGORY;
    }
    
    public int getObjectPriority(Object object)
    {
        if (CartHelper.getCartsNode(_resolver).equals(object))
        {
            return 10;
        }
        return super.getObjectPriority(object);
    }
}
