/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.cart.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.cart.Cart;
import org.ametys.plugins.cart.Cart.CartProfile;
import org.ametys.plugins.cart.CartsDAO;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * SAX carts
 *
 */
public class GetCartsAction extends ServiceableAction
{
    private CurrentUserProvider _userProvider;
    private AmetysObjectResolver _resolver;
    private CartsDAO _cartDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _cartDAO = (CartsDAO) serviceManager.lookup(CartsDAO.ROLE);
        _userProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String profileId = StringUtils.defaultIfEmpty((String) jsParameters.get("profile"), "read_access");
        
        List<Map<String, Object>> nodes = new ArrayList<>();
        
        if (!"read_access".equals(profileId) && !"write_access".equals(profileId) && !"right_access".equals(profileId))
        {
            throw new IllegalArgumentException("Unexpected profile identifier : " + profileId);
        }
        CartProfile profile = CartProfile.valueOf(profileId.toUpperCase());

        UserIdentity identity = _userProvider.getUser();
        Predicate<Cart> filterCarts;
        switch (profile)
        {
            case WRITE_ACCESS:
                filterCarts = cart -> _cartDAO.canWrite(identity, cart);
                break;
            case RIGHT_ACCESS:
                filterCarts = cart -> _cartDAO.canAssignRights(identity, cart);
                break;
            case READ_ACCESS:
            default:
                filterCarts = cart -> _cartDAO.canRead(identity, cart);
                break;
        }

        nodes = _resolver.query("//element(*, ametys:cart)")
                        .stream()
                        .filter(Cart.class::isInstance)
                        .map(obj -> (Cart) obj)
                        .filter(filterCarts)
                        .map(cart -> _cartDAO.getCartProperties(cart))
                        .collect(Collectors.toList());
                
        
        Map<String, Object> result = new HashMap<>();
        result.put("carts", nodes);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
}
