/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.cart.generators;

import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.cart.Cart;
import org.ametys.plugins.cart.CartElement;
import org.ametys.plugins.cart.CartsDAO;
import org.ametys.plugins.cart.ContentElement;
import org.ametys.plugins.cart.QueryFromDirectoryElement;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.version.VersionAwareAmetysObject;
import org.ametys.plugins.workflow.store.AmetysStep;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.spi.Step;

/**
 * SAX a {@link Cart}
 *
 */
public class CartElementsGenerator extends ServiceableGenerator
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;
    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;
    /** The current user provider */
    protected CurrentUserProvider _userProvider;
    /** The user manager */
    protected UserManager _userManager;
    /** The carts DAO */
    protected CartsDAO _cartsDAO;
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _cTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
        _workflowHelper = (WorkflowHelper) smanager.lookup(WorkflowHelper.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _userProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _cartsDAO = (CartsDAO) smanager.lookup(CartsDAO.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String id = request.getParameter("cartId");
        
        Cart cart = _resolver.resolveById(id);
        
        contentHandler.startDocument();
        
        UserIdentity user = _userProvider.getUser();
        if (_cartsDAO.canRead(user, cart))
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", cart.getId());
            
            XMLUtils.startElement(contentHandler, "cart", attrs);
            
            XMLUtils.createElement(contentHandler, "label", cart.getTitle());
            XMLUtils.createElement(contentHandler, "description", cart.getDescription());
            
            List<CartElement> elements = cart.getElements();
            for (CartElement elmt : elements)
            {
                _saxElement(elmt);
            }
            
            XMLUtils.createElement(contentHandler, "total", String.valueOf(elements.size()));
            
            XMLUtils.endElement(contentHandler, "cart");
        }
        else
        {
            XMLUtils.createElement(contentHandler, "not-allowed");
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * SAX an element of the cart
     * @param elmt The element to SAX
     * @throws SAXException if something goes wrong when saxing the element
     */
    protected void _saxElement (CartElement elmt) throws SAXException
    {
        List<I18nizableText> groups = elmt.getGroups();
        
        for (I18nizableText group : groups)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", elmt.getId());
            
            XMLUtils.startElement(contentHandler, "cartElement", attrs);
                    
            elmt.getTitle().toSAX(contentHandler, "title");
            elmt.getDescription().toSAX(contentHandler, "description");
            group.toSAX(contentHandler, "group");
            
            XMLUtils.createElement(contentHandler, "creation", DateUtils.zonedDateTimeToString(elmt.getCreationDate()));
            
            UserIdentity creatorIdentity = elmt.getCreator();
            User creator = _userManager.getUser(creatorIdentity);
            XMLUtils.createElement(contentHandler, "creator", creator != null ? creator.getFullName() : StringUtils.EMPTY);
            
            XMLUtils.createElement(contentHandler, "lastModification", DateUtils.zonedDateTimeToString(elmt.getLastModified()));
            
            UserIdentity lastContributorIdentity = elmt.getLastContributor();
            User lastContributor = _userManager.getUser(lastContributorIdentity);
            XMLUtils.createElement(contentHandler, "lastContributor", lastContributor != null ? lastContributor.getFullName() : StringUtils.EMPTY);
            
            String glyphIcon = elmt.getGlyphIcon();
            if (glyphIcon != null)
            {
                XMLUtils.createElement(contentHandler, "iconGlyph", glyphIcon);
            }
            
            String smallIcon = elmt.getSmallIcon();
            if (smallIcon != null)
            {
                XMLUtils.createElement(contentHandler, "smallIcon", smallIcon);
            }
            String mediumIcon = elmt.getSmallIcon();
            if (mediumIcon != null)
            {
                XMLUtils.createElement(contentHandler, "mediumIcon", mediumIcon);
            }
            
            XMLUtils.createElement(contentHandler, "type", elmt.getType());
            
            if (elmt instanceof ContentElement)
            {
                Content content = _resolver.resolveById(elmt.getId());
                _saxWorkflowStep(content);
            }
            else if (elmt instanceof QueryFromDirectoryElement queryElement)
            {
                XMLUtils.createElement(contentHandler, "documentation", queryElement.getQuery().getDocumentation());
                XMLUtils.createElement(contentHandler, "queryType", queryElement.getQuery().getType());
            }
            
            XMLUtils.endElement(contentHandler, "cartElement");
        }
    }

    /**
     * SAX the workflow step if the content is a <code>WorkflowAwareContent</code>
     * @param content The content
     * @throws SAXException if an error occurs while SAXing.
     */
    protected void _saxWorkflowStep (Content content) throws SAXException
    {
        if (content instanceof WorkflowAwareContent)
        {
            WorkflowAwareContent waContent = (WorkflowAwareContent) content;
            
            try
            {
                AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow(waContent);
                long workflowId = waContent.getWorkflowId();
                String workflowName = workflow.getWorkflowName(workflowId);
                
                Step currentStep = _getCurrentStep(waContent, workflow);
                
                int currentStepId = currentStep.getStepId();
                
                I18nizableText workflowStepName = new I18nizableText("application",  _workflowHelper.getStepName(workflowName, currentStepId));
                
                AttributesImpl atts = new AttributesImpl();
                atts.addAttribute("", "id", "id", "CDATA", String.valueOf(currentStepId));
                if ("application".equals(workflowStepName.getCatalogue()))
                {
                    atts.addAttribute("", "icon-small", "icon-small", "CDATA", "/plugins/cms/resources_workflow/" + workflowStepName.getKey() + "-small.png");
                    atts.addAttribute("", "icon-medium", "icon-medium", "CDATA", "/plugins/cms/resources_workflow/" + workflowStepName.getKey() + "-medium.png");
                    atts.addAttribute("", "icon-large", "icon-large", "CDATA", "/plugins/cms/resources_workflow/" + workflowStepName.getKey() + "-large.png");
                }
                else
                {
                    String pluginName = workflowStepName.getCatalogue().substring("plugin.".length());
                    atts.addAttribute("", "icon-small", "icon-small", "CDATA", "/plugins/" + pluginName + "/resources/img/workflow/" + workflowStepName.getKey() + "-small.png");
                    atts.addAttribute("", "icon-medium", "icon-medium", "CDATA", "/plugins/" + pluginName + "/resources/img/workflow/" + workflowStepName.getKey() + "-medium.png");
                    atts.addAttribute("", "icon-large", "icon-large", "CDATA", "/plugins/" + pluginName + "/resources/img/workflow/" + workflowStepName.getKey() + "-large.png");
                }
                
                XMLUtils.startElement(contentHandler, "workflow-step", atts);
                workflowStepName.toSAX(contentHandler);
                XMLUtils.endElement(contentHandler, "workflow-step");
            }
            catch (AmetysRepositoryException e)
            {
                // Current step id was not positioned
            }
            catch (WorkflowException e)
            {
                // Ignore, just don't SAX the workflow step.
            }
        }
    }
    
    /**
     * Get a content's step, wherever it works on the base version or not.
     * @param content the content.
     * @param workflow the workflow instance bound to this content
     * @return the content's workflow step.
     * @throws WorkflowException if an error occurs.
     */
    protected Step _getCurrentStep(WorkflowAwareContent content, AmetysObjectWorkflow workflow) throws WorkflowException
    {
        long workflowId = content.getWorkflowId();
        Step currentStep = (Step) workflow.getCurrentSteps(workflowId).get(0);
        
        if (content instanceof VersionAwareAmetysObject)
        {
            VersionAwareAmetysObject vaContent = (VersionAwareAmetysObject) content;
            String currentRevision = vaContent.getRevision();
            
            if (currentRevision != null)
            {
                
                String[] allRevisions = vaContent.getAllRevisions();
                int currentRevIndex = ArrayUtils.indexOf(allRevisions, currentRevision);
                
                if (currentRevIndex > -1 && currentRevIndex < (allRevisions.length - 1))
                {
                    String nextRevision = allRevisions[currentRevIndex + 1];
                    
                    Date currentRevTimestamp = vaContent.getRevisionTimestamp();
                    Date nextRevTimestamp = vaContent.getRevisionTimestamp(nextRevision);
                    
                    // Get all steps between the two revisions.
                    List<Step> steps = _workflowHelper.getStepsBetween(workflow, workflowId, currentRevTimestamp, nextRevTimestamp);
                    
                    // In the old workflow structure
                    // We take the second, which is current revision's last step.
                    if (steps.size() > 0 && steps.get(0) instanceof AmetysStep)
                    {
                        AmetysStep amStep = (AmetysStep) steps.get(0);
                        if (amStep.getProperty("actionFinishDate") != null)
                        {
                            // New workflow structure detected: cut the first workflow step
                            // in the list, as it belongs to the next version.
                            steps = steps.subList(1, steps.size());
                        }
                    }
                    
                    // Order by step descendant.
                    Collections.sort(steps, new Comparator<Step>()
                    {
                        public int compare(Step step1, Step step2)
                        {
                            return -Long.valueOf(step1.getId()).compareTo(step2.getId());
                        }
                    });
                    
                    // The first step in the list is the current version's last workflow step.
                    if (steps.size() > 0)
                    {
                        currentStep = steps.get(0);
                    }
                }
            }
        }
        return currentStep;
    }
}
