/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.classified.ads;

import java.time.Period;
import java.time.ZonedDateTime;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.trace.ForensicLogger;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.DateExpression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.runtime.config.Config;

/**
 * A {@link Schedulable} job which delete classified ads after an defined number of days
 */
public class DeleteClassifiedAdsSchedulable extends AbstractStaticSchedulable
{
    /** The key for the number of month */
    public static final String JOBDATAMAP_NB_MONTH = "nbMonth";
    
    /** The content type id */
    public static final String CLASSIFIED_ADS_CONTENT_TYPE = "org.ametys.plugins.classified.ads.Content.ads";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content DAO */
    protected ContentDAO _contentDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentDAO = (ContentDAO) manager.lookup(ContentDAO.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        JobDataMap jobDataMap = context.getJobDetail().getJobDataMap();
        Long delay = (Long) jobDataMap.get(Scheduler.PARAM_VALUES_PREFIX + JOBDATAMAP_NB_MONTH);
        if (delay == null)
        {
            delay = Config.getInstance().<Long>getValue("classified-ads.conservation.delay", false, null);
        }
        
        if (delay == null || delay < 0)
        {
            throw new IllegalArgumentException("The expiration delay must be positive");
        }
        
        ZonedDateTime expirationDate = ZonedDateTime.now().minus(Period.ofMonths(delay.intValue()));
        AndExpression filter = new AndExpression();
        filter.add(new DateExpression(DefaultContent.METADATA_CREATION, Operator.LE, DateUtils.asDate(expirationDate), ExpressionContext.newInstance().withInternal(true)));
        filter.add(new ContentTypeExpression(Operator.EQ, CLASSIFIED_ADS_CONTENT_TYPE));
        
        String query = ContentQueryHelper.getContentXPathQuery(filter);
        
        try (AmetysObjectIterable<Content> contents = _resolver.query(query))
        {
            _contentDAO.forceDeleteContentsObj(contents.stream().toList(), null);
            ForensicLogger.info("data.policy.classified-ads", Map.of("handled", Long.toString(contents.getSize())), UserPopulationDAO.SYSTEM_USER_IDENTITY);
        }
    }
}
