/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/**
 * DAO for collections of synchronizable contents.
 */
Ext.define('Ametys.cms.contentio.SynchronizableContentsCollectionDAO', {
    singleton: true,
    
    constructor: function(config)
    {
        /**
         * @callable
         * @member Ametys.cms.contentio.SynchronizableContentsCollectionDAO
         * @method getEditionConfiguration 
         * Gets the configuration for creating/editing a collection of synchronizable contents.
         * This calls the method 'getEditionConfiguration' of the server DAO 'SynchronizableContentsCollectionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO",
                methodName: "getEditionConfiguration",
                errorMessage: {
                    msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_GET_EDITION_CONFIGURATION_ERROR}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.cms.contentio.SynchronizableContentsCollectionDAO
         * @method getCollectionParameterValues 
         * Gets the values of the parameters of the given collection
         * This calls the method 'getCollectionParameterValues' of the server DAO 'SynchronizableContentsCollectionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the collection
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO",
                methodName: "getCollectionParameterValues",
                callback: {
                    handler: this._getCollectionParameterValuesCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_GET_PARAMETER_VALUES_ERROR}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.cms.contentio.SynchronizableContentsCollectionDAO
         * @method getCollection 
         * Adds a new collection of synchronizable contents
         * This calls the method 'getSynchronizableContentsCollectionAsJson' of the server DAO 'SynchronizableContentsCollectionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.collectionId The id of the synchronizable contents collection to get
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO",
                methodName: "getSynchronizableContentsCollectionAsJson",
                localName: "getCollection",
                errorMessage: {
                    msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_GET_ERROR}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.cms.contentio.SynchronizableContentsCollectionDAO
         * @method addCollection 
         * Adds a new collection of synchronizable contents
         * This calls the method 'addCollection' of the server DAO 'SynchronizableContentsCollectionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.values The collection parameters values
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO",
                methodName: "addCollection",
                callback: {
                    handler: this._addCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_ADD_ERROR}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.cms.contentio.SynchronizableContentsCollectionDAO
         * @method editCollection 
         * Edits the given collection of synchronizable contents
         * This calls the method 'editCollection' of the server DAO 'SynchronizableContentsCollectionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the collection
         * @param {String} parameters.label The label of the collection
         * @param {String} parameters.contentTypeId The id of the content type to create when populating the collection
         * @param {String} parameters.modelId The id of the model of the SynchronizableContentsCollection
         * @param {Object} parameters.parameters The parameters
         * @param {Boolean} parameters.synchronizeExistingContentsOnly True to synchronize only existing contents
         * @param {Boolean} parameters.removalSync True for removal sync
         * @param {Boolean} parameters.ignoreRestrictions True to ignore restrictions on synchronization
         * @param {Boolean} parameters.checkCollection True to check collection on synchronization
         * @param {String[]} parameters.compatibleSCC List of compatible collections
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO",
                methodName: "editCollection",
                callback: {
                    handler: this._editCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_EDIT_ERROR}}",
                    category:Ext.getClassName(this)
                }
            }
        );
        
        /**
         * @callable
         * @member Ametys.cms.contentio.SynchronizableContentsCollectionDAO
         * @method removeCollection 
         * Removes the given collection
         * This calls the method 'removeCollection' of the server DAO 'SynchronizableContentsCollectionDAO'.
         * @param {Object[]} parameters The parameters to transmit to the server method
         * @param {String} parameters.id The id of the collection to remove
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
            {
                role: "org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO",
                methodName: "removeCollection",
                callback: {
                    handler: this._removeCb
                },
                errorMessage: {
                    msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_REMOVE_ERROR}}",
                    category:Ext.getClassName(this)
                }
            }
        );
    },
    
    /**
     * @private
     * Callback function called after {@link #getCollectionParameterValues} has been processed.
     * @param {Object} response The server response
     */
    _getCollectionParameterValuesCb: function(response)
    {
        var error = response['error'] || '';
        if (error == "unknown")
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_ERROR_UNKNOWN_TITLE}}",
                msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_ERROR_UNKNOWN}}",
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.ERROR
            });
        }
    },
    
    /**
     * @private
     * Callback function called after {@link #addCollection} has been processed.
     * @param {String} id The id of the created collection
     */
    _addCb: function(id)
    {
        if (id != null)
        {
            Ext.create('Ametys.message.Message', {
                type: Ametys.message.Message.CREATED,
                targets: [{
                    id: Ametys.message.MessageTarget.SYNCHRONIZABLE_CONTENTS_COLLECTION,
                    parameters: {
                        id: id
                    }
                }]
            });
        }
        else
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_ADD_ERROR}}",
                msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_ADD_ERROR}}",
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.ERROR
            });
        }
    },
    
    /**
     * @private
     * Callback function called after {@link #editCollection} has been processed.
     * @param {Object} response The server response
     */
    _editCb: function(response)
    {
        var error = response['error'] || '';
        var id = response['id'];
        if (error == "unknown")
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_ERROR_UNKNOWN_TITLE}}",
                msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_ERROR_UNKNOWN}}",
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.ERROR
            });
        }
        else if (id != null)
        {
            Ext.create('Ametys.message.Message', {
                type: Ametys.message.Message.MODIFIED,
                targets: [{
                    id: Ametys.message.MessageTarget.SYNCHRONIZABLE_CONTENTS_COLLECTION,
                    parameters: {
                        id: id
                    }
                }]
            });
        }
    },
    
    /**
     * @private
     * Callback function called after {@link #removeCollection} has been processed.
     * @param {Object} response The server response
     */
    _removeCb: function(response)
    {
        var error = response['error'] || '';
        var id = response['id'];
        if (error == "unknown")
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_ERROR_UNKNOWN_TITLE}}",
                msg: "{{i18n PLUGINS_CONTENTIO_COLLECTION_DAO_ERROR_UNKNOWN}}",
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.ERROR
            });
        }
        else if (id != null)
        {
            Ext.create('Ametys.message.Message', {
                type: Ametys.message.Message.DELETED,
                targets: [{
                    id: Ametys.message.MessageTarget.SYNCHRONIZABLE_CONTENTS_COLLECTION,
                    parameters: {
                        id: id
                    }
                }]
            });
        }
    }
});
