/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.archive;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;

import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.util.AmetysHomeHelper;

import com.fasterxml.jackson.core.JsonParseException;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;
import com.google.gson.reflect.TypeToken;

/**
 * Archive data on disk.
 */
public class ArchiveHandler extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** Avalon role. */
    public static final String ROLE = ArchiveHandler.class.getName();
    
    /** The prefix path for exporting metadata content */
    public static final String METADATA_PREFIX = "_metadata/";
    
    static final String MANIFEST_FILENAME = "manifest.json";
    
    private static final Gson __GSON = new GsonBuilder()
            .setPrettyPrinting()
            .disableHtmlEscaping()
            .create();
    
    /** The context */
    protected Context _context;
    
    private ArchiverExtensionPoint _archiverEP;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _archiverEP = (ArchiverExtensionPoint) manager.lookup(ArchiverExtensionPoint.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Gets the folder containing all the managed archives
     * @return the folder containing all the managed archives
     */
    public Path getArchiveFolder()
    {
        Path allArchives = AmetysHomeHelper.getAmetysHomeData().toPath().resolve("archives");
        allArchives.toFile().mkdirs();
        return allArchives;
    }
    
    /**
     * Gets the archive files
     * @return the archive files
     */
    public Stream<String> getArchiveFiles()
    {
        Path archiveFolder = getArchiveFolder();
        return FileUtils.listFiles(archiveFolder.toFile(), new String[] {"zip"}, false)
                .stream()
                .map(File::getName);
    }
    
    /**
     * Gets the archive file
     * @param name The archive name
     * @return the archive file
     */
    public File getArchiveFile(String name)
    {
        if (!FilenameUtils.isExtension(name, "zip"))
        {
            throw new IllegalArgumentException(String.format("Wrong archive name '%s', it should be a ZIP file", name));
        }
        
        Path archiveFolder = getArchiveFolder();
        File archive = archiveFolder.resolve(name)
                .toFile();
        
        if (archive.exists())
        {
            return archive;
        }
        
        throw new IllegalArgumentException(String.format("Archive file '%s' does not exist", archive));
    }
    
    /**
     * Gets the partial imports of the given archive
     * @param archiveName The archive name
     * @return the partial imports
     * @throws IOException if an I/O error occured
     */
    public Stream<PartialImport> getPartialImports(String archiveName) throws IOException
    {
        Objects.requireNonNull(archiveName, "The archive name cannot be null");
        
        Path archiveFolder = getArchiveFolder();
        Path archive = archiveFolder.resolve(archiveName);
        return _partialImports(archive);
    }
    
    private Stream<PartialImport> _partialImports(Path archive) throws IOException
    {
        try (InputStream manifestIs = ZipEntryHelper.zipEntryFileInputStream(archive, MANIFEST_FILENAME);)
        {
            String manifestContent = IOUtils.toString(manifestIs, StandardCharsets.UTF_8);
            Type mapType = new TypeToken<Map<String, Object>>()
                { /*empty*/ }
                .getType();
            Map<String, Object> manifestJson = __GSON.fromJson(manifestContent, mapType);
            return _partialImportsFromJson(manifestJson);
        }
        catch (JsonParseException | JsonSyntaxException e)
        {
            throw new IOException(String.format("An error occured with the manifest file '%s' of archive '%s'.\nThe cause is: %s", MANIFEST_FILENAME, archive, e.getMessage()));
        }
    }
    
    private Stream<PartialImport> _partialImportsFromJson(Map<String, Object> manifestJson)
    {
        return _archiverEP.getExtensionsIds()
                .stream()
                .filter(manifestJson::containsKey)
                .map(archiverId -> _partialImportsFromJsonAndArchiver(manifestJson, archiverId))
                .flatMap(Function.identity());
    }
    
    private Stream<PartialImport> _partialImportsFromJsonAndArchiver(Map<String, Object> manifestJson, String archiverId)
    {
        Archiver archiver = _archiverEP.getExtension(archiverId);
        ManifestReaderWriter manifestReaderWriter = archiver.getManifestReaderWriter();
        Object jsonData = manifestJson.get(archiverId);
        return manifestReaderWriter.toPartialImports(jsonData);
    }
    
    /**
     * Exports with {@link ArchiverExtensionPoint} mechanism into the given {@link File}
     * @param output The output ZIP file
     * @throws IOException if an IO exception occured
     */
    public void export(File output) throws IOException
    {
        try (var fileOs = new FileOutputStream(output);
             var bufferedOs = new BufferedOutputStream(fileOs);
             var zos = new ZipOutputStream(bufferedOs, StandardCharsets.UTF_8))
        {
            // Handle manifest data
            _exportManifest(zos);
            
            // Process actual data
            _exportActualData(zos);
        }
    }
    
    private void _exportManifest(ZipOutputStream zos) throws IOException
    {
        Map<String, Object> manifestData = new LinkedHashMap<>();
        
        for (String id : _archiverEP.getExtensionsIds())
        {
            Archiver archiver = _archiverEP.getExtension(id);
            ManifestReaderWriter manifestReaderWriter = archiver.getManifestReaderWriter();
            Object archiverManifestData = manifestReaderWriter.getData();
            manifestData.put(id, archiverManifestData);
        }
        
        String json = __GSON.toJson(manifestData);
        ZipEntry manifestEntry = new ZipEntry(MANIFEST_FILENAME);
        zos.putNextEntry(manifestEntry);
        IOUtils.write(json, zos, StandardCharsets.UTF_8);
    }
    
    private void _exportActualData(ZipOutputStream zos) throws IOException
    {
        for (String id : _archiverEP.getExtensionsIds())
        {
            Archiver archiver = _archiverEP.getExtension(id);
            
            getLogger().info("Processing archiver {} ({}) for export", id, archiver.getClass().getName());

            archiver.export(zos);
        }
    }
    
    private ImportReport _partialImport(String archiverId, Path zipPath, Collection<String> partialImports, Merger merger) throws IOException
    {
        Archiver archiver = _archiverEP.getExtension(archiverId);
        Collection<String> managedPartialImports = archiver.managedPartialImports(partialImports);
        if (managedPartialImports.isEmpty())
        {
            getLogger().info("Archiver '{}' ({}) not processed, given partial imports ({}) do not match", archiverId, archiver.getClass().getName(), partialImports);
            return new ImportReport();
        }
        else
        {
            getLogger().info("Processing archiver '{}' ({}) for import, with matching partial imports {}", archiverId, archiver.getClass().getName(), managedPartialImports);
            return archiver.partialImport(zipPath, managedPartialImports, merger, merger.deleteBeforePartialImport());
        }
    }
    
    /**
     * Imports all data from the given ZIP archive
     * @param input the ZIP archive
     * @param merger the {@link Merger}
     * @return The {@link ImportReport}
     * @throws IOException if an I/O error occurs
     */
    public ImportReport importAll(File input, Merger merger) throws IOException
    {
        Path zipPath = input.toPath();
        // Import all declared partial imports
        Collection<String> allPartialImports = _partialImports(zipPath)
                .map(PartialImport::getKey)
                .collect(Collectors.toList());
        return partialImport(input, allPartialImports, merger);
    }
    
    /**
     * Imports partial data from the given ZIP archive
     * @param input the ZIP archive
     * @param partialImports the partial imports
     * @param merger the {@link Merger}
     * @return The {@link ImportReport}
     * @throws IOException if an I/O error occurs
     */
    public ImportReport partialImport(File input, Collection<String> partialImports, Merger merger) throws IOException
    {
        List<ImportReport> reports = new ArrayList<>();
        
        Path zipPath = input.toPath();
        for (String id : _archiverEP.getExtensionsIds())
        {
            ImportReport partialImportResult = _partialImport(id, zipPath, partialImports, merger);
            reports.add(partialImportResult);
        }
        
        return ImportReport.union(reports);
    }
}
