/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.csv;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.cocoon.servlet.multipart.PartOnDisk;
import org.apache.cocoon.servlet.multipart.RejectedPart;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.supercsv.prefs.CsvPreference;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.servlet.RuntimeConfig;

/**
 * Import contents from an uploaded CSV file.
 */
public class ImportCSVFile implements Serviceable, Contextualizable, Component
{
    /** Avalon Role */
    public static final String ROLE = ImportCSVFile.class.getName();
    
    /** The ametys home folder where csv files are stored before import */
    static final String CONTENTIO_STORAGE_DIRECTORY = "contentio/temp"; // FIXME should be relate to AmetysTmp not AmetysHome CONTENTIO-334
    
    private static final String[] _ALLOWED_EXTENSIONS = new String[] {"txt", "csv"};
    
    private ContentTypeExtensionPoint _contentTypeEP;
    
    private WorkflowHelper _workflowHelper;
    
    private ImportCSVFileHelper _importCSVFileHelper;

    private Enumerator _synchronizeModeEnumerator;
    
    private Context _context;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _contentTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _workflowHelper = (WorkflowHelper) serviceManager.lookup(WorkflowHelper.ROLE);
        _importCSVFileHelper = (ImportCSVFileHelper) serviceManager.lookup(ImportCSVFileHelper.ROLE);
        _synchronizeModeEnumerator = (Enumerator) serviceManager.lookup(SynchronizeModeEnumerator.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Import CSV file
     * @param escapingChar The escaping character
     * @param separatingChar The separating character
     * @param contentTypeId The content type id
     * @param part The CSV file's Part
     * @return The result of the import
     * @throws Exception If an error occurs
     */
    @Callable (rights = "Plugins_ContentIO_ImportFile", context = "/cms")
    public Map importCSVFile(String escapingChar, String separatingChar, String contentTypeId, Part part) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        
        Map<String, Object> result = new HashMap<>();
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);

        if (part instanceof RejectedPart || part == null)
        {
            result.put("success", false);
            result.put("error", "rejected");
        }
        else
        {
            PartOnDisk uploadedFilePart = (PartOnDisk) part;
            File uploadedFile = uploadedFilePart.getFile();
            
            String filename = uploadedFilePart.getFileName().toLowerCase();
            
            if (!FilenameUtils.isExtension(filename, _ALLOWED_EXTENSIONS))
            {
                result.put("error", "invalid-extension");
                request.setAttribute(JSonReader.OBJECT_TO_READ, result);
                return result;
            }
            
            try (
                InputStream fileInputStream = new FileInputStream(uploadedFile);
                InputStream inputStream = IOUtils.buffer(fileInputStream);
            )
            {
                Charset charset = _importCSVFileHelper.detectCharset(inputStream);
                result.put("charset", charset);
                String[] headers = _importCSVFileHelper.extractHeaders(inputStream, _getCSVPreference(escapingChar, separatingChar), charset);
              
                if (headers == null)
                {
                    result.put("error", "no-header");
                    request.setAttribute(JSonReader.OBJECT_TO_READ, result);
                    return result;
                }
                
                List<Map<String, Object>> mapping = _importCSVFileHelper.getMapping(contentType, headers);
                result.put("mapping", mapping);
                
                // Build a map to count how many attribute are there for each group
                Map<String, Long> attributeCount = mapping.stream()
                        .map(map -> map.get(ImportCSVFileHelper.MAPPING_COLUMN_ATTRIBUTE_PATH))
                        .map(String.class::cast)
                        .filter(StringUtils::isNotEmpty)
                        .map(attributePath -> {
                            String attributePrefix = "";
                            int endIndex = attributePath.lastIndexOf("/");
                            if (endIndex != -1)
                            {
                                attributePrefix = attributePath.substring(0, endIndex);
                            }
                            return attributePrefix;
                        })
                        .collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));
                
                // If an attribute is the only for its group, it is the identifier
                for (Map<String, Object> map : mapping)
                {
                    String attributePath = (String) map.get(ImportCSVFileHelper.MAPPING_COLUMN_ATTRIBUTE_PATH);
                    String attributePrefix = StringUtils.EMPTY;
                    int endIndex = attributePath.lastIndexOf("/");
                    if (endIndex != -1)
                    {
                        attributePrefix = attributePath.substring(0, endIndex);
                    }
    
                    boolean parentIsContent;
                    // If there is no prefix, it is an attribute of the content we want to import
                    if (attributePrefix.equals(StringUtils.EMPTY))
                    {
                        parentIsContent = true;
                    }
                    // Otherwise, check the modelItem
                    else if (contentType.hasModelItem(attributePrefix))
                    {
                        ModelItem modelItem = contentType.getModelItem(attributePrefix);
                        String modelTypeId = modelItem.getType().getId();
                        parentIsContent = ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID.equals(modelTypeId);
                    }
                    else
                    {
                        parentIsContent = false;
                    }
                    
                    // If an attribute is the only one of its level, and is part of a content, consider it as the identifier
                    if (attributeCount.getOrDefault(attributePrefix, 0L).equals(1L) && parentIsContent && StringUtils.isNotBlank(attributePath))
                    {
                        map.put(ImportCSVFileHelper.MAPPING_COLUMN_IS_ID, true);
                    }
                }
                
                String importId = UUID.randomUUID().toString();
                _copyFile(uploadedFile.toPath(), importId);
                result.put("importId", importId);
                result.put("success", true);
                result.put("workflows", _getWorkflows());
                result.put("defaultWorkflow", contentType.getDefaultWorkflowName().orElse(null));
                result.put("fileName", uploadedFile.getName());
                
                List<Map<String, Object>> synchronizeModes = ((Map<String, I18nizableText>) _synchronizeModeEnumerator.getEntries())
                        .entrySet()
                        .stream()
                        .map(entry -> Map.of("value", entry.getKey(), "label", entry.getValue()))
                        .toList();
                result.put("synchronizeModes", synchronizeModes);
            }
        }
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return result;
    }
    
    private void _copyFile(Path path, String importId) throws IOException
    {
        File contentIOStorageDir = FileUtils.getFile(RuntimeConfig.getInstance().getAmetysHome(), CONTENTIO_STORAGE_DIRECTORY);
        
        if (!contentIOStorageDir.exists())
        {
            if (!contentIOStorageDir.mkdirs())
            {
                throw new IOException("Unable to create monitoring directory: " + contentIOStorageDir);
            }
        }
        Files.copy(path, Paths.get(contentIOStorageDir.getPath(), importId + ".csv"), StandardCopyOption.REPLACE_EXISTING);
    }
    
    private CsvPreference _getCSVPreference(String escapingChar, String separatingChar)
    {
        char separating = separatingChar.charAt(0);
        char escaping = escapingChar.charAt(0);
        
        if (separating == escaping)
        {
            throw new IllegalArgumentException("Separating character can not be equals to escaping character");
        }
        return new CsvPreference.Builder(escaping, separating, "\r\n").build();
    }
    
    /**
     * getWorkflows
     * @return map of workflows
     */
    private List<Map<String, Object>> _getWorkflows()
    {
        List<Map<String, Object>> workflows = new ArrayList<>();
        String[] workflowNames = _workflowHelper.getWorkflowNames();
        for (String workflowName : workflowNames)
        {
            Map<String, Object> workflowMap = new HashMap<>();
            workflowMap.put("value", workflowName);
            workflowMap.put("label", _workflowHelper.getWorkflowLabel(workflowName));
            workflows.add(workflowMap);
        }
        return workflows;
    }
}
