/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.rights;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for retrieving root of synchronized contents
 */
public class SynchronizedRootContentHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The component role. */
    public static final String ROLE = SynchronizedRootContentHelper.class.getName();
    
    /** The root node name of the plugin */
    public static final String CONTENTIO_ROOT_NODE = "contentio";
    
    /** The root node name of the imported contents */
    public static final String IMPORTED_CONTENTS_ROOT_NODE = RepositoryConstants.NAMESPACE_PREFIX + ":contents";
    
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Gets the root of contents
     * @param create <code>true</code> to create automatically the root when missing.
     * @return the root of contents
     * @throws UnknownAmetysObjectException If root node does not exist
     */
    public AmetysObjectCollection getRootContent(boolean create)
    {
        ModifiableTraversableAmetysObject pluginsNode = _resolver.resolveByPath("/ametys:plugins/");
        
        boolean needSave = false;
        if (!pluginsNode.hasChild(CONTENTIO_ROOT_NODE))
        {
            if (create)
            {
                pluginsNode.createChild(CONTENTIO_ROOT_NODE, "ametys:unstructured");
                needSave = true;
            }
            else
            {
                throw new UnknownAmetysObjectException("Node '/ametys:plugins/contentio' is missing");
            }
        }
        
        ModifiableTraversableAmetysObject contentIoNode = pluginsNode.getChild(CONTENTIO_ROOT_NODE);
        if (!contentIoNode.hasChild(IMPORTED_CONTENTS_ROOT_NODE))
        {
            if (create)
            {
                contentIoNode.createChild(IMPORTED_CONTENTS_ROOT_NODE, "ametys:collection");
                needSave = true;
            }
            else
            {
                throw new UnknownAmetysObjectException("Node '/ametys:plugins/contentio/ametys:contents' is missing");
            }
        }
        
        if (needSave)
        {
            pluginsNode.saveChanges();
        }
        
        return contentIoNode.getChild(IMPORTED_CONTENTS_ROOT_NODE);
    }
    
    /**
     * Tests if the given content is child of the root of contents, i.e. the given content is managed by the CMS workspace
     * @param content The content to test
     * @return true if the given content is child of the root of contents
     */
    public boolean isChildOfRootContent(Content content)
    {
        try
        {
            AmetysObjectCollection rootContent = getRootContent(false);
            
            AmetysObject parent = content.getParent();
            while (parent != null)
            {
                if (parent.equals(rootContent))
                {
                    return true;
                }
                
                parent = parent.getParent();
            }
            
            return false;
        }
        catch (UnknownAmetysObjectException e)
        {
            return false;
        }
    }
}
