/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.contentio.synchronize.search;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.ServerCommHelper;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO;

/**
 * Action to search in a SCC.
 */
public class SCCSearchAction extends ServiceableAction
{
    /** Field imported to display in SCCSearchTool */
    protected static final String FIELD_IMPORTED = "imported";
    
    /** SCC DAO */
    protected SynchronizableContentsCollectionDAO _synchronizableContentsCollectionDAO;
    /** ServerComm Helper */
    protected ServerCommHelper _serverCommHelper;
    /** JSON Utils */
    protected JSONUtils _jsonUtils;
    
    private Logger _logger = LoggerFactory.getLogger(SCCSearchAction.class);
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _synchronizableContentsCollectionDAO = (SynchronizableContentsCollectionDAO) smanager.lookup(SynchronizableContentsCollectionDAO.ROLE);
        _serverCommHelper = (ServerCommHelper) smanager.lookup(ServerCommHelper.ROLE);
        _jsonUtils = (JSONUtils) smanager.lookup(JSONUtils.ROLE);
    }

    @SuppressWarnings("unchecked")
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        // Get JS parameters
        Map<String, Object> jsParameters = _serverCommHelper.getJsParameters();
        
        // Mask imported
        Map<String, Object> searchParams = (Map<String, Object>) jsParameters.get("values");
        boolean maskImported = searchParams.containsKey(SCCSearchModelConfiguration.CRITERIA_MASK_IMPORTED) && Boolean.valueOf(searchParams.get(SCCSearchModelConfiguration.CRITERIA_MASK_IMPORTED).toString());
        searchParams.remove(SCCSearchModelConfiguration.CRITERIA_MASK_IMPORTED);
        
        // Get the collection for search
        String collectionId = (String) jsParameters.get("collectionId");
        SynchronizableContentsCollection collection = _synchronizableContentsCollectionDAO.getSynchronizableContentsCollection(collectionId);
        
        // Search
        Integer offset = _getIntValue(jsParameters, "start", 0);
        Integer limit = _getIntValue(jsParameters, "limit", Integer.MAX_VALUE);

        // Construct the JSON object
        Map<String, Object> objectToRead = new HashMap<>();
        
        // Different treatment if we need to mask imported or not (one is optimized)
        if (maskImported)
        {
            Map<String, Map<String, Object>> results = collection.search(searchParams, 0, Integer.MAX_VALUE, _getSortList(jsParameters.get("sort")), _logger);
            Iterator<String> resultKeys = results.keySet().iterator();
            
            // Skip offset
            int currentOffset = 0;
            while (currentOffset < offset && resultKeys.hasNext())
            {
                if (collection.getContent(null, resultKeys.next(), true) == null)
                {
                    currentOffset++;
                }
            }
            
            // Get the contents to the limit
            List<Map<String, Object>> finalResults = new LinkedList<>();
            int currentLimit = 0;
            while (currentLimit < limit && resultKeys.hasNext())
            {
                String idValue = resultKeys.next();
                Map<String, Object> contentResults = new HashMap<>(results.get(idValue));
                if (collection.getContent(null, idValue, true) == null)
                {
                    // Add to results
                    contentResults.put(FIELD_IMPORTED, false);
                    finalResults.add(contentResults);
                    currentLimit++;
                }
            }
            
            objectToRead.put("items", finalResults);
            // FIXME CONTENTIO-87
            objectToRead.put("total", collection.getTotalCount(searchParams, _logger));
        }
        else
        {
            Map<String, Map<String, Object>> results = new LinkedHashMap<>(collection.search(searchParams, offset, limit, _getSortList(jsParameters.get("sort")), _logger));

            for (String idValue : results.keySet())
            {
                Map<String, Object> contentResults = new HashMap<>(results.get(idValue));
                contentResults.put(FIELD_IMPORTED, collection.getContent(null, idValue, true) != null);
                results.put(idValue, contentResults);
            }
            
            objectToRead.put("items", results.values());
            objectToRead.put("total", collection.getTotalCount(searchParams, _logger));
        }
        
        // Add JSON to the request to be parsed
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, objectToRead);
        
        return EMPTY_MAP;
    }
    
    private int _getIntValue(Map<String, Object> values, String key, int defaultValue)
    {
        if (values.containsKey(key))
        {
            return Integer.valueOf(values.get(key).toString()).intValue();
        }
        
        return defaultValue;
    }
    
    private List<Object> _getSortList(Object sortValues)
    {
        if (sortValues != null)
        {
            return _jsonUtils.convertJsonToList(sortValues.toString());
        }
        
        return null;
    }
}
