/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.datafiller;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.LocaleUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.observation.ObservationManager;
import org.ametys.plugins.classified.ads.ClassifiedAdsDataPolicy;
import org.ametys.plugins.repository.metadata.MultilingualString;
import org.ametys.plugins.ugc.page.UGCPageHandler;
import org.ametys.runtime.model.View;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.sitemap.Sitemap;

import com.opensymphony.workflow.WorkflowException;

/**
 * Create and add generic content in test pages to test render, skins, and services
 */
public class ClassifiedAdsContentCreationManager extends AbstractLogEnabled implements Component, Serviceable
{
    /** Avalon Role */
    public static final String ROLE = ClassifiedAdsContentCreationManager.class.getName();

    private static final String __CLASSIFICATION_ATTRIBUTE = "category";
    private static final String __CATEGORY_CONTENT = "org.ametys.plugins.classified.ads.Content.categories";

    /** The UGC page handler */
    protected UGCPageHandler _ugcPageHandler;
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The content type extension point */
    private ContentTypeExtensionPoint _cTypeEP;

    /** The Sitemap creation Manager */
    private SitemapPopulator _sitemapPopulator;

    private PageDAO _pageDAO;

    private ContentCreationHelper _contentCreationHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentCreationHelper = (ContentCreationHelper) manager.lookup(ContentCreationHelper.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _sitemapPopulator = (SitemapPopulator) manager.lookup(SitemapPopulator.ROLE);
        _ugcPageHandler = (UGCPageHandler) manager.lookup(UGCPageHandler.ROLE);
    }
    
    /**
     * Create test pages filled with generic content for test purposes
     * @param sitemap the sitemap where the content will be inserted
     * @throws Exception when the page creation fails
     */
    public void createClassifiedAdsPages(Sitemap sitemap) throws Exception
    {
        // create classified ads category entries
        ContentType categContentType = _cTypeEP.getExtension(__CATEGORY_CONTENT);
        Map<String, Object> categValues = _getValuesForContentType(categContentType);
        MultilingualString titles = new MultilingualString();
        titles.add(LocaleUtils.toLocale("fr"), "Ameublement");
        categValues.put("title", titles);
        Content ameublement = _createContent(sitemap, "ameublement", categContentType, categValues);
        titles.add(LocaleUtils.toLocale("fr"), "Immobilier");
        Content immobilier = _createContent(sitemap, "immobilier", categContentType, categValues);
        titles.add(LocaleUtils.toLocale("fr"), "Mode");
        Content mode = _createContent(sitemap, "mode", categContentType, categValues);
        
        // create classified ads root page
        ModifiablePage rootPage = _sitemapPopulator.getOrCreatePage(sitemap, "Petites annonces");
        try
        {
            _pageDAO.setTemplate(List.of(rootPage.getId()), "page");
        }
        catch (IllegalStateException e)
        {
            getLogger().warn("Failed to set template for page {}", rootPage.getTitle(), e);
        }
        
        _ugcPageHandler.setUGCRoot(rootPage, ClassifiedAdsDataPolicy.CLASSIFIED_ADS_CONTENT_TYPE, __CLASSIFICATION_ATTRIBUTE, true);
        
        // create ads
        ContentType contentType = _cTypeEP.getExtension(ClassifiedAdsDataPolicy.CLASSIFIED_ADS_CONTENT_TYPE);
        Map<String, Object> values = _getValuesForContentType(contentType);
        values.put("category", ameublement);
        values.put("title", "Table basse design en bois massif et pieds en fonte style industriel");
        _createContent(sitemap, "table-basse-petite-annonce", contentType, values);
        values.put("category", immobilier);
        values.put("title", "Location Chalet tout équipé avec parking jusqu'à 8 personnes");
        _createContent(sitemap, "location-chalet-petite-annonce", contentType, values);
        values.put("category", mode);
        values.put("title", "Baskets édition limitée taille 39");
        _createContent(sitemap, "baskets-petite-annonce", contentType, values);
        values.put("title", "T-shirt bleu chiné neuf");
        _createContent(sitemap, "t-shirt-petite-annonce", contentType, values);
    }
    
    private Content _createContent(Sitemap sitemap, String contentName, ContentType contentType, Map<String, Object> values) throws WorkflowException
    {
        WorkflowAwareContent content = _contentCreationHelper.getContent(contentName, sitemap);
        if (content == null)
        {
            content = _contentCreationHelper.initializeContent(contentType, sitemap, contentName);
            content = _contentCreationHelper.editContent(content, contentType, values);
        }
        return content;
    }
    
    private Map<String, Object> _getValuesForContentType(ContentType contentType)
    {
        View view = org.ametys.runtime.model.ViewHelper.getTruncatedView(View.of(contentType));
        Map<String, Object> values = _contentCreationHelper.getGenericValuesForView(view);
        return values;
    }
}
