/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The DataSource DAO.
 */
 Ext.define('Ametys.cms.externaldata.QueryDAO', {
 	singleton: true,
 	
 	constructor: function(config)
 	{
		/**
		 * @callable
		 * @member Ametys.cms.externaldata.QueryDAO
		 * @method addQuery 
		 * Adds a query.
		 * This calls the method 'addQuery' of the server DAO 'QueryDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
 		 * @param {String} parameters.siteName The site name
		 * @param {Object} parameters.params The params needed to create the query, at least :
		 * @param {String} parameters.params.type The type of the datasource associated to this query
		 * @param {String} parameters.params.name The name of the query
		 * @param {String} parameters.params.description The description of the query
		 * @param {String} parameters.params.resultType Can only be "MULTIPLE" or "SIMPLE"
		 * @param {String} parameters.params.dataSourceId The id of the datasource
		 * @param {Object} parameters.additionalparameters Additional parameters needed to create the datasource
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.externaldata.data.QueryDao",
				methodName: "addQuery",
	     		callback: {
	         		handler: this._addQueryCb
	     		},
				errorMessage: {
				    msg: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_ADD_QUERY_ERROR}}",
				    category: Ext.getClassName(this)
				}
			}
		);
		
		/**
		 * @callable
		 * @member Ametys.cms.externaldata.QueryDAO
		 * @method updateQuery 
		 * Updates a query.
		 * This calls the method 'updateQuery' of the server DAO 'QueryDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
 		 * @param {String} parameters.siteName The site name
		 * @param {Object} parameters.params The params needed to create the query, at least :
		 * @param {String} parameters.params.id The id of the query to edit
		 * @param {String} parameters.params.type The type of the datasource associated to this query
		 * @param {String} parameters.params.name The name of the query
		 * @param {String} parameters.params.description The description of the query
		 * @param {String} parameters.params.resultType Can only be "MULTIPLE" or "SIMPLE"
		 * @param {String} parameters.params.dataSourceId The id of the datasource
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
				role: "org.ametys.plugins.externaldata.data.QueryDao",
	     		methodName: "updateQuery",
	     		callback: {
	         		handler: this._updateQueryCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_EDIT_QUERY_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
			}
		);
		/**
		 * @callable
		 * @member Ametys.cms.externaldata.QueryDAO
		 * @method deleteQuery 
		 * Deletes a query.
		 * This calls the method 'deleteQuery' of the server DAO 'QueryDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.siteName The site name
		 * @param {String} parameters.id The id of the query to delete
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.externaldata.data.QueryDao",
	     		methodName: "deleteQuery",
	     		callback: {
	         		handler: this._deleteQueryCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_DELETE_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.externaldata.QueryDAO
		 * @method getQueryProperties 
		 * Retrieves a query properties.
		 * This calls the method 'getQueryProperties' of the server DAO 'QueryDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the datasource to retrieve
		 * @param {String} parameters.siteName The site name
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
				role: "org.ametys.plugins.externaldata.data.QueryDao",
	     		methodName: "getQueryProperties",
	     		callback: {
					handler: Ext.emptyFn,
					scope: this
				},
				errorMessage: {
					msg: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_GET_QUERY_ERROR}}",
					category: Ext.getClassName(this)
				}
			}
		);
 	},
 	
 	/**
 	 * Callback function called after {@link #addQuery} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_addQueryCb: function(response)
 	{
		var error = response['error'];
		if (error == 'unknown-type')
		{
			Ametys.Msg.show({
				title: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_ADD_QUERY}}",
				msg: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_ADD_QUERY_UNKNOWNTYPE}}",
				buttons: Ext.Msg.OK,
				icon: Ext.MessageBox.ERROR
			});
		}
		else if (error == 'add-query-error')
		{
			Ametys.Msg.show({
				title: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_ADD_QUERY_ERROR_TITLE}}",
				msg: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_ADD_QUERY_ERROR}}",
				buttons: Ext.Msg.OK,
				icon: Ext.MessageBox.ERROR
			});
		}
		else
		{
	    	Ext.create("Ametys.message.Message", {
	    		type: Ametys.message.Message.CREATED,
	    		targets: {
	    		    id: Ametys.message.MessageTarget.EXTERNAL_DATASOURCE_QUERY,
	    			parameters: {
	    				id: response['id'],
	    				parentId: response['parentId'],
	    				dataSourceType: response['type']
	    			}
	    		}
	    	});
		}
 	},
 	
 	/**
 	 * Callback function called after {@link #updateQuery} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_updateQueryCb: function(response)
 	{
		var error = response['error'];
		if (error == 'update-query-error')
        {
            Ametys.Msg.show({
                   title: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_EDIT_QUERY_ERROR_TITLE}}",
                   msg: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_EDIT_QUERY_ERROR}}",
                   buttons: Ext.Msg.OK,
                   icon: Ext.MessageBox.ERROR
            });
        }
        else
        {
			Ext.create("Ametys.message.Message", {
				type: Ametys.message.Message.MODIFIED,
				targets: {
				    id: Ametys.message.MessageTarget.EXTERNAL_DATASOURCE_QUERY,
					parameters: {
						id: response['id'],
						dataSourceType: response['type']
					}
				}
			});
        }
 	},
 	
 	/**
 	 * Callback function called after {@link #deleteQuery} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_deleteQueryCb: function(response)
 	{
	    var error = response['error'];
	    if (error == 'delete-query-error')
	    {
	        Ametys.Msg.show({
	            title: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_DELETE_ERROR_TITLE}}",
	            msg: "{{i18n PLUGINS_EXTERNAL_DATA_HANDLE_QUERIES_DELETE_ERROR}}",
	            buttons: Ext.Msg.OK,
	            icon: Ext.MessageBox.ERROR
	        });
	    }
	    else
	    {
		    Ext.create("Ametys.message.Message", {
		    	type: Ametys.message.Message.DELETED,
		    	targets: {
		    	    id: Ametys.message.MessageTarget.EXTERNAL_DATASOURCE_QUERY, 
		        	parameters: {
		        		id: response['id']
		        	}
		        }
		    });
	    }
	 }
 	
 });