/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;

import org.apache.commons.lang3.StringUtils;

/**
 * An abstract query.<br>
 * Defines getters and setters for all base members of a query, and provides
 * some helpful methods.
 */
public abstract class AbstractQuery implements Query
{
    
    /** The query id. */
    protected String _id;
    
    /** The query name. */
    protected String _name;
    
    /** The query description. */
    protected String _description;
    
    /** The query parameter names. */
    protected Map<String, String> _parameters;
    
    /** The query result type. */
    protected ResultType _resultType;
    
    /** The datasource id. */
    protected String _dataSourceId;
    
    /** The id of source factory */
    protected String _factoryId;
    
    @Override
    public String getId()
    {
        return _id;
    }
    
    /**
     * Set the query id.
     * @param id the id of the query
     */
    public void setId(String id)
    {
        this._id = id;
    }
    
    @Override
    public String getName()
    {
        return _name;
    }
    
    /**
     * Set the query name.
     * @param name the name of the query
     */
    public void setName(String name)
    {
        this._name = name;
    }
    
    @Override
    public String getDescription()
    {
        return _description;
    }
    
    /**
     * Set the query description.
     * @param description the description of the query
     */
    public void setDescription(String description)
    {
        this._description = description;
    }
    
    @Override
    public ResultType getResultType()
    {
        return _resultType;
    }
    
    /**
     * Set the query result type.
     * @param resultType the query result type.
     */
    public void setResultType(ResultType resultType)
    {
        this._resultType = resultType;
    }
    
    @Override
    public String getFactory()
    {
        return _factoryId;
    }
    
    /**
     * Set the data source factory
     * @param factoryId The id of source factory
     */
    public void setFactory (String factoryId)
    {
        _factoryId = factoryId;
    }
    
    @Override
    public String getDataSourceId()
    {
        return _dataSourceId;
    }
    
    /**
     * Set the id of data source
     * @param dataSourceId The id of data source
     */
    public void setDataSourceId (String dataSourceId)
    {
        _dataSourceId = dataSourceId;
    }
    
    /**
     * Extract parameters from a query string in the form ${parameterName}.
     * @param query the query containing parameters.
     * @return the parameter names as a Set.
     */
    protected Map<String, String> _buildParameters(String query)
    {
        Map<String, String> params = new HashMap<>();
        
        if (query != null)
        {
            Matcher matcher = Query.PARAMETER_MATCHER.matcher(query);
            List<String> paramWithLabel = new ArrayList<>();
            while (matcher.find())
            {
                if (matcher.groupCount() == 1)
                {
                    // Convert to lowercase to avoid browser problems.
                    String paramName = matcher.group(1).toLowerCase();
                    if (!params.containsKey(paramName) || !paramWithLabel.contains(paramName))
                    {
                        params.put(paramName, paramName);
                    }
                }
                else if (matcher.groupCount() > 2)
                {
                    // Convert to lowercase to avoid browser problems.
                    String paramName = matcher.group(1).toLowerCase();
                    String label = matcher.group(3);
                    if (StringUtils.isBlank(label))
                    {
                        label = paramName;
                        paramWithLabel.add(paramName);
                    }
                    
                    if (!params.containsKey(paramName) || !paramWithLabel.contains(paramName))
                    {
                        params.put(paramName, label);
                    }
                }
            }
        }
        
        return params;
    }
    
}
