/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data;

import java.io.IOException;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Generator for external data search service
 *
 */
public class ExternalSearchGenerator extends ServiceableGenerator
{
    private static final String __SEARCH_CRITERIA_PREFIX = "external-data-search-";
    
    /** The Query DAO. */
    protected QueryDao _queryDao;
    /** Datasource factory handler */
    protected DataSourceFactoryExtensionPoint _dataSourceFactoryEP;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _queryDao = (QueryDao) smanager.lookup(QueryDao.ROLE);
        _dataSourceFactoryEP = (DataSourceFactoryExtensionPoint) smanager.lookup(DataSourceFactoryExtensionPoint.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String currentSiteName = null;
        String lang = null;
        Page page = (Page) request.getAttribute(Page.class.getName());
        if (page != null)
        {
            currentSiteName = page.getSiteName();
            lang = page.getSitemapName();
        }
        
        String siteName = (String) request.getAttribute("site");
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());
        
        String idQuery = zoneItem.getServiceParameters().getValue("datasource-query");
        int limit = Math.toIntExact(zoneItem.getServiceParameters().getValue("limit"));
        int pageNum = getPageIndex(request);
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "search");
        
        // The search url
        XMLUtils.createElement(contentHandler, "url", page != null ? lang + "/" + page.getPathInSitemap() + ".html" : lang + "/_plugins/" + currentSiteName + "/" + lang + "/service/search.html");

        try
        {
            Query query = _queryDao.getQuery(siteName, idQuery);
            Map<String, String> parameterValues = getParameterValues(request, query);
            
            saxFormParameters (request, query, parameterValues);
            
            boolean submit = request.getParameter("external-data-search-submit") != null;
            if (submit)
            {
                
                XMLUtils.startElement(contentHandler, "search-result");
                executeQuery(query, parameterValues, limit, pageNum);
                XMLUtils.endElement(contentHandler, "search-result");
            }
            
        }
        catch (DataInclusionException e)
        {
            getLogger().error("An error occurred : impossible to get query with id : " + idQuery, e);
        }
        XMLUtils.endElement(contentHandler, "search");
        contentHandler.endDocument();
    }
    
    /**
     * Get the search criteria values from request
     * @param request The request
     * @param query The LDAP query
     * @return the values
     */
    protected Map<String, String> getParameterValues (Request request, Query query)
    {
        Map<String, String> paramValues = new HashMap<>();
        
        Map<String, String> parametersName = query.getParameters();
        for (String paramName : parametersName.keySet())
        {
            String value = request.getParameter(__SEARCH_CRITERIA_PREFIX + paramName); 
            if (StringUtils.isNotBlank(value))
            {
                paramValues.put(paramName, value);
            }
        }
        return paramValues;
    }
    
    /**
     * SAX the search parameters from the request parameters and query
     * @param request The request
     * @param query The LDAP query
     * @param values The extracted form values
     * @throws SAXException If an error occurs while SAXing
     */
    protected void saxFormParameters (Request request, Query query, Map<String, String> values)  throws SAXException
    {
        XMLUtils.startElement(contentHandler, "form");
        
        XMLUtils.startElement(contentHandler, "fields");
        saxFormFields (query);
        XMLUtils.endElement(contentHandler, "fields");
        
        boolean submit = request.getParameter("external-data-search-submit") != null;
        if (submit)
        {
            XMLUtils.startElement(contentHandler, "values");
            saxFormValues(request, values);
            XMLUtils.endElement(contentHandler, "values");
        }
        
        XMLUtils.endElement(contentHandler, "form");
    }
    
    /**
     * SAX the form search criteria
     * @param query The LDAP query
     * @throws SAXException if an error occurs while SAXing
     */
    protected void saxFormFields (Query query) throws SAXException
    {
        Map<String, String> parametersName = query.getParameters();
        for (Entry<String, String> entry : parametersName.entrySet())
        {
            String parameterName = entry.getKey();
            String paramaterLabel = entry.getValue();
            AttributesImpl att = new AttributesImpl();
            att.addCDATAAttribute("id", parameterName);

            XMLUtils.createElement(contentHandler, "field", att, paramaterLabel);
        }
    }
    
    /**
     * SAX the form search criteria values
     * @param request The request
     * @param values The extracted form values
     * @throws SAXException if an error occurs while SAXing
     */
    protected void saxFormValues (Request request, Map<String, String> values) throws SAXException
    {
        for (Entry<String, String> entry : values.entrySet())
        {
            String paramName = entry.getKey();
            String paramValue = entry.getValue();
            
            XMLUtils.createElement(contentHandler, paramName, paramValue);
        }
    }
    
    /** 
     * Execute ldap query
     * @param query the ldap query
     * @param parameterValues the search criteria
     * @param nbResultsPerPage the number of result per page
     * @param page the page number
     * @throws DataInclusionException if an error occurred
     * @throws SAXException If an error occurred
     */
    protected void executeQuery(Query query, Map<String, String> parameterValues, int nbResultsPerPage, int page) throws DataInclusionException, SAXException
    {
        String factoryId = query.getFactory();
        DataSourceFactory<Query, QueryResult> dsFactory = _dataSourceFactoryEP.getExtension(factoryId);
        
        int offset = (page - 1) * nbResultsPerPage;
        
        QueryResult result = dsFactory.execute(query, parameterValues, offset, nbResultsPerPage);
        
        saxResult(query, result);
        
        boolean hasMoreElmts = false;
        int nbResults = result.getSize();
        if (nbResults >= nbResultsPerPage)
        {
            // Test if there has more elements
            hasMoreElmts = dsFactory.execute(query, parameterValues, page * nbResultsPerPage, 1).getSize() > 0;
        }
        
        saxPagination(page, offset, nbResultsPerPage, result.getSize(), hasMoreElmts);
    }
    
    /**
     * SAX the pagination
     * @param page the current page
     * @param offset the offset
     * @param limit the number of results per page
     * @param nbResult the number result of the page
     * @param hasMoreResult true if there are less one page more
     * @throws SAXException if an error occurred
     */
    protected void saxPagination(int page, int offset, int limit, int nbResult, boolean hasMoreResult) throws SAXException
    {
        AttributesImpl atts = new AttributesImpl();
        atts.addCDATAAttribute("start", String.valueOf(offset));
        atts.addCDATAAttribute("end", String.valueOf(offset + nbResult));
        XMLUtils.startElement(contentHandler, "pagination", atts);
        
        int nbPage = page;
        if (hasMoreResult)
        {
            nbPage++;
        }
        
        for (int i = 1; i <= nbPage; i++)
        {
            AttributesImpl attPage = new AttributesImpl();
            attPage.addCDATAAttribute("index", String.valueOf(i));
            attPage.addCDATAAttribute("start", String.valueOf((i - 1) * limit));
            XMLUtils.createElement(contentHandler, "page", attPage);
        }
        
        XMLUtils.endElement(contentHandler, "pagination");
    }

    /**
     * Sax a query result.
     * @param query the query.
     * @param result the result to generate.
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxResult(Query query, QueryResult result) throws SAXException
    {
        String queryId = query.getId();
        try
        {
            // Others enhancement handlers should not touch the generated HTML
            contentHandler.processingInstruction("ametys-unmodifiable", "start");
            
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("class", "data " + queryId);
            XMLUtils.startElement(contentHandler, "table", atts);
            Collection<String> colNames = result.getColumnNames();
            
            if (!colNames.isEmpty())
            {
                XMLUtils.startElement(contentHandler, "tr");
                for (String columnName : colNames)
                {
                    XMLUtils.createElement(contentHandler, "th", columnName);
                }
                XMLUtils.endElement(contentHandler, "tr");
            }
            
            int index = 0;
            for (QueryResultRow row : result)
            {
                AttributesImpl attr = new AttributesImpl();
                if (index % 2 != 0)
                {
                    attr.addAttribute("", "class", "class", "CDATA", "odd");
                }
                else
                {
                    attr.addAttribute("", "class", "class", "CDATA", "even");
                }
                XMLUtils.startElement(contentHandler, "tr", attr);
                
                for (String columnName : colNames)
                {
                    String value = row.get(columnName);
                    XMLUtils.createElement(contentHandler, "td", StringUtils.defaultString(value));
                }
                XMLUtils.endElement(contentHandler, "tr");
                index++;
            }
            
            XMLUtils.endElement(contentHandler, "table");
            
            contentHandler.processingInstruction("ametys-unmodifiable", "end");
        }
        catch (DataInclusionException e)
        {
            String message = "Unable to get the query results (query ID : " + queryId + ")";
            getLogger().error(message, e);
            throw new SAXException(message, e);
        }
        finally
        {
            result.close();
        }
    }
    /**
     * Get the page index
     * @param request The request
     * @return The page index
     */
    protected int getPageIndex (Request request)
    {
        Enumeration paramNames = request.getParameterNames();
        while (paramNames.hasMoreElements())
        {
            String param = (String) paramNames.nextElement();
            if (param.startsWith("page-"))
            {
                return Integer.parseInt(param.substring("page-".length()));
            }
        }
        return 1;
    }
}
