/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data;

import java.util.Map;
import java.util.regex.Pattern;

import org.ametys.core.datasource.DataSourceClientInteraction.DataSourceType;

/**
 * &lt;p&gt;Data interrogation query class.&lt;/p&gt;
 * &lt;p&gt;Defines an interrogation query (SQL select, LDAP search, ...) on a specified
 * data source, with optional parameters, which returns either a single result
 * (i.e. name of a person in a LDAP directory), or a list of multiple structured
 * results (i.e. list of products from a SQL database).
 * &lt;/p&gt;
 */
public interface Query
{
    /** Parameter pattern. */
    public static final String PARAMETER_PATTERN = "\\$\\{(\\w+)(\\[([^\\]]+)\\])?\\}";
    
    /** Parameter matcher. */
    public static final Pattern PARAMETER_MATCHER = Pattern.compile(PARAMETER_PATTERN);
    
    /** Query result type. */
    public enum ResultType
    {
        /** Simple result : single (scalar) value. */
        SIMPLE,
        /** Multiple result : table of structured values. */
        MULTIPLE;
    }
    
    /**
     * Get the query ID.
     * @return the query ID.
     */
    String getId();
    
    /**
     * Get the query name.
     * @return the query name.
     */
    String getName();
    
    /**
     * Get the query description.
     * @return the query description.
     */
    String getDescription();
    
    /**
     * Get the query type.
     * @return the query type.
     */
    DataSourceType getType();
    
    /**
     * Get the id of source factory
     * @return the id of source factory
     */
    String getFactory();
    
    /**
     * Get the query parameter names.
     * @return the query parameter names as a Collection.
     */
    Map<String, String> getParameters();
    
    /**
     * Get the query result type.
     * @return the query result type.
     */
    ResultType getResultType();
    
    /**
     * Get the additional query configuration information.
     * @return the additional query configuration information as a Map of String -&gt; String.
     */
    Map<String, String> getAdditionalConfiguration();
    
    /**
     * Get the id of datasource on which depends this query.
     * @return the datasource on which depends this query.
     */
    String getDataSourceId();
}
