/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data.ldap;

import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.datasource.DataSourceClientInteraction.DataSourceType;
import org.ametys.plugins.externaldata.data.AbstractQuery;

/**
 * A LDAP query.
 */
public class LdapQuery extends AbstractQuery
{
    /** Attribute pattern (matches attributeName[attributeLabel]). */
    public static final String ATTRIBUTE_PATTERN = "(\\w+)(?:\\[([^\\]]+)\\])?";
    
    /** Attribute matcher. */
    public static final Pattern ATTRIBUTE_MATCHER = Pattern.compile(ATTRIBUTE_PATTERN);
    
    /** The LDAP attributes to return. */
    protected String _attributes;
    
    /** The LDAP constraint, optionally contains parameters. */
    protected String _constraint;
    
    /** Relative DN for users. */
    protected String _relativeDN;
    
    /** The LDAP constraint, optionally contains parameters. */
    protected String _scope;
    
    /** The decoded attributes. */
    protected Map<String, String> _attributeMap;
    
    @Override
    public DataSourceType getType()
    {
        return DataSourceType.LDAP;
    }
    
    /**
     * Get the LDAP attributes to return.
     * @return the LDAP attributes.
     */
    public String getAttributes()
    {
        return _attributes;
    }
    
    /**
     * Set the LDAP attributes to return.
     * @param attributes LDAP attributes to return.
     */
    public void setAttributes(String attributes)
    {
        this._attributes = attributes;
    }
    
    /**
     * Get the LDAP constraint.
     * @return the LDAP constraint.
     */
    public String getConstraint()
    {
        return _constraint;
    }
    
    /**
     * Set the LDAP constraint.
     * @param constraint the LDAP constraint to set.
     */
    public void setConstraint(String constraint)
    {
        this._constraint = constraint;
    }
    
    /**
     * Set relative DN for users
     * @param relativeDN the LDAP relative DN for users
     */
    public void setRelativeDN(String relativeDN)
    {
        this._relativeDN = relativeDN;
    }
    
    /**
     * Get the relative DN for users
     * @return the relative DN for users
     */
    public String getRelativeDN()
    {
        return _relativeDN;
    }
    
    /**
     * Get the LDAP scope.
     * @return the LDAP scope
     */
    public String getScope()
    {
        return _scope;
    }

    /**
     * Set the LDAP scope.
     * @param scope the LDAP scope to set
     */
    public void setScope(String scope)
    {
        _scope = scope;
    }
    
    @Override
    public Map<String, String> getParameters()
    {
        if (_parameters == null)
        {
            _parameters = _buildParameters(_constraint);
        }
        return _parameters;
    }
    
    @Override
    public Map<String, String> getAdditionalConfiguration()
    {
        Map<String, String> additionalConf = new HashMap<>();
        
        additionalConf.put("attributes", _attributes);
        additionalConf.put("constraint", _constraint);
        additionalConf.put("scope", _scope);
        additionalConf.put("relativeDN", _relativeDN);
        
        return additionalConf;
    }
    
    /**
     * Get attributes as Map.
     * @return the attributes as a Map of attributeName -&gt; label.
     */
    public Map<String, String> getAttributesAsMap()
    {
        if (_attributeMap == null)
        {
            _attributeMap = _buildAttributesMap(_attributes);
        }
        return _attributeMap;
    }
    
    /**
     * Extract attributes from a query string in the form "attr1[label1], attr2[label2], attr3".
     * @param query the query containing parameters.
     * @return the attributes as a Map of label -&gt; attributeName. If no label was provided between brackets, the name is used.
     */
    protected Map<String, String> _buildAttributesMap(String query)
    {
        Map<String, String> attributes = new LinkedHashMap<>();
        
        if (query != null)
        {
            String[] attributeArray = query.split(",");
            
            for (String attribute : attributeArray)
            {
                String attributeTrimmed = attribute.trim();
                Matcher matcher = ATTRIBUTE_MATCHER.matcher(attributeTrimmed);
                while (matcher.find())
                {
                    if (matcher.groupCount() == 1)
                    {
                        String attrName = matcher.group(1);
                        attributes.put(attrName, attrName);
                    }
                    else if (matcher.groupCount() > 1)
                    {
                        String attrName = matcher.group(1);
                        String attrLabel = matcher.group(2);
                        if (StringUtils.isBlank(attrLabel))
                        {
                            attrLabel = attrName;
                        }
                        attributes.put(attrLabel, attrName);
                    }
                }
            }
        }
        
        return attributes;
    }
}
