/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data.sql;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.plugins.externaldata.data.DataInclusionException;
import org.ametys.plugins.externaldata.data.QueryResult;
import org.ametys.plugins.externaldata.data.QueryResultRow;
import org.ametys.plugins.externaldata.data.Query.ResultType;

/**
 * SQL query result.
 */
public class LazySqlQueryResult implements QueryResult, Iterator<QueryResultRow>
{
    
    /** The result type. */
    protected ResultType _resultType;
    
    /** The wrapped ResultSet */
    protected ResultSet _resultSet;
    
    /** The column names. */
    private Collection<String> _columnNames;
    
    /**
     * Constructs a query result around a {@link ResultSet}.
     * @param resultSet the result set
     */
    public LazySqlQueryResult(ResultSet resultSet)
    {
        _resultSet = resultSet;
        try
        {
            _columnNames = _extractColumnNames(_resultSet);
        }
        catch (SQLException e)
        {
            throw new RuntimeException(e);
        }
    }
    
    @Override
    public ResultType getType()
    {
        return _resultType;
    }
    
    @Override
    public int getSize()
    {
        try
        {
            return _resultSet.getFetchSize();
        }
        catch (SQLException e)
        {
            return -1;
        }
    }
    
    /**
     * Set the result type.
     * @param resultType the result type.
     */
    public void setType(ResultType resultType)
    {
        this._resultType = resultType;
    }
    
    @Override
    public Collection<String> getColumnNames() throws DataInclusionException
    {
        try
        {
            if (_columnNames == null)
            {
                _columnNames = _extractColumnNames(_resultSet);
            }
            return _columnNames;
        }
        catch (SQLException e)
        {
            throw new DataInclusionException("Unable to retrieve the column names.", e);
        }
    }
    
    @Override
    public Iterator<QueryResultRow> iterator()
    {
        return this;
    }

    @Override
    public boolean hasNext()
    {
        try
        {
            return _resultSet.next();
        }
        catch (SQLException e)
        {
            // TODO Throw a RuntimeException.
            throw new RuntimeException(e);
        }
    }

    @Override
    public QueryResultRow next()
    {
        try
        {
            SqlQueryResultRow row = new SqlQueryResultRow();
            
            for (String colName : getColumnNames())
            {
                String value = _resultSet.getString(colName);
                row.put(colName, value);
            }
            
            return row;
        }
        catch (DataInclusionException e)
        {
            throw new RuntimeException(e);
        }
        catch (SQLException e)
        {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void remove()
    {
        throw new UnsupportedOperationException("Remove operation not supported.");
    }
    
    @Override
    public void close()
    {
        ConnectionHelper.cleanup(_resultSet);
    }
    
    /**
     * Extract the column names of a JDBC result set.
     * @param resultSet the JDBC result set.
     * @return the column names as a Collection.
     * @throws SQLException if an exception occurs when manipulating the result set
     */
    protected Collection<String> _extractColumnNames(ResultSet resultSet) throws SQLException
    {
        ResultSetMetaData rsMeta = resultSet.getMetaData();
        int count = rsMeta.getColumnCount();
        
        List<String> columns = new ArrayList<>(count);
        
        for (int i = 1; i <= count; i++)
        {
            columns.add(rsMeta.getColumnLabel(i));
        }
        
        return columns;
    }
    
}
