/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.externaldata.data.sql;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.ametys.plugins.externaldata.data.DataInclusionException;
import org.ametys.plugins.externaldata.data.QueryResult;
import org.ametys.plugins.externaldata.data.QueryResultRow;
import org.ametys.plugins.externaldata.data.Query.ResultType;

/**
 * SQL query result.
 */
public class SqlQueryResult implements QueryResult
{
    
    /** The result type. */
    protected ResultType _resultType;
    
    /** The result rows. */
    protected List<QueryResultRow> _rows;
    
    /** The column names. */
    private Collection<String> _columnNames;
    
    /**
     * Constructs a query result around a ResultSet.
     * @param resultSet the result set
     */
    public SqlQueryResult(ResultSet resultSet)
    {
        try
        {
            _extractResults(resultSet);
        }
        catch (SQLException e)
        {
            throw new RuntimeException(e);
        }
    }
    
    @Override
    public ResultType getType()
    {
        return _resultType;
    }
    
    @Override
    public int getSize()
    {
        return _rows.size();
    }
    
    /**
     * Set the result type.
     * @param resultType the result type.
     */
    public void setType(ResultType resultType)
    {
        this._resultType = resultType;
    }
    
    @Override
    public Collection<String> getColumnNames() throws DataInclusionException
    {
        return _columnNames;
    }
    
    @Override
    public Iterator<QueryResultRow> iterator()
    {
        return _rows.iterator();
    }
    
    @Override
    public void close()
    {
        // Does nothing.
    }
    
    /**
     * Extract the column names and results from a JDBC result set.
     * @param resultSet the JDBC result set.
     * @throws SQLException if an error occurs when manipulating the result set
     */
    protected void _extractResults(ResultSet resultSet) throws SQLException
    {
        ResultSetMetaData rsMeta = resultSet.getMetaData();
        int count = rsMeta.getColumnCount();
        
        // First, extract the column names.
        _columnNames = new ArrayList<>(count);
        for (int i = 1; i <= count; i++)
        {
            String columnLabel = rsMeta.getColumnLabel(i);
            // Exclude 'rnum' which is used for pagination of Oracle results
            if (!"rnum".equals(columnLabel.toLowerCase())) 
            {
                _columnNames.add(rsMeta.getColumnLabel(i));
            }
        }
        
        // Then, extract the results.
        _rows = new ArrayList<>();
        while (resultSet.next())
        {
            SqlQueryResultRow row = new SqlQueryResultRow();
            
            for (String colName : _columnNames)
            {
                String value = resultSet.getString(colName);
                row.put(colName, value);
            }
            
            _rows.add(row);
        }
    }
    
}
