/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.authentication.oidc.endofauthenticationprocess;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;

import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.site.Site;
import org.ametys.plugins.site.SiteInformationCache.SignupPage;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.exception.ServiceUnavailableException;
import org.ametys.site.BackOfficeRequestHelper;

/**
 * Entry point of the end of the authentication process
 */
public class TemporarySignup extends EndOfAuthenticationProcess implements Serviceable
{
    /** The user manager. */
    private JSONUtils _jsonUtils;

    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
    }

    private String _buildRedirectURI(Request request, String redirectPage, String email, String firstName, String lastName, String token)
    {
        // creation of redirect URI (The URI the issuer (google, fb etc) is going to redirect to)
        StringBuilder uriBuilder = new StringBuilder();
        uriBuilder.append(request.getScheme() + "://").append(request.getServerName());

        if (request.isSecure())
        {
            if (request.getServerPort() != 443)
            {
                uriBuilder.append(":");
                uriBuilder.append(request.getServerPort());
            }
        }
        else
        {
            if (request.getServerPort() != 80)
            {
                uriBuilder.append(":");
                uriBuilder.append(request.getServerPort());
            }
        }

        uriBuilder.append(request.getContextPath());
        uriBuilder.append("/_extra-user-management/temp-signup-callback");

        String actualRedirectUri = uriBuilder.toString();
        String redirectUri = redirectPage;

        // If the temporary sign-up worked
        if (token != null)
        {
            // Add the parameters email and token
            redirectUri += "?email=" + email;
            redirectUri += "&token=" + token; 

            // If they are not null add the parameters first and last names to pre-fill the fields
            if (firstName != null)
            {
                redirectUri += "&firstname=" + URIUtils.encodeParameter(firstName);
            }

            if (lastName != null)
            {
                redirectUri += "&lastname=" + URIUtils.encodeParameter(lastName);
            }
        }

        actualRedirectUri += "?redirectPage=" + URIUtils.encodeParameter(redirectUri);

        return actualRedirectUri;
    }

    private String _buildRequest(String cmsUrl, String siteName, String populationId, String userDirectoryId, String language, String email)
    {
        Map<String, String> parameters = Map.of("siteName", siteName, "populationID", populationId, "userDirectoryID", userDirectoryId, "language", language, "email", email);
        String requestUri = URIUtils.encodeURI(cmsUrl + "/plugins/web/user-signup/tempSignup", parameters);
        
        return requestUri;
    }

    private String _temporarySignup(String siteName, String populationId, String userDirectoryId, String language, String email) throws IllegalStateException, ServiceUnavailableException
    {
        // Create request to send to the back to temporary sign-in the user
        String cmsUrl = Config.getInstance().getValue("org.ametys.site.bo");
        String token = null;

        try (CloseableHttpClient httpClient = BackOfficeRequestHelper.getHttpClient())
        {
            HttpGet httpGet = new HttpGet(_buildRequest(cmsUrl, siteName, populationId, userDirectoryId, language, email));
            // Add header to attest that the request come from the front office
            httpGet.addHeader("X-Ametys-FO", "true");
            // Add header to attest that the request come from the front office web application and not from a user
            httpGet.addHeader("X-Ametys-Server", "true");

            try (CloseableHttpResponse response = httpClient.execute(httpGet); ByteArrayOutputStream os = new ByteArrayOutputStream())
            {
                switch (response.getStatusLine().getStatusCode())
                {
                    case 200: 
                        try (InputStream stream = response.getEntity().getContent()) 
                        {
                            // Read all data from URL
                            String providerInfo = IOUtils.toString(stream, StandardCharsets.UTF_8);
                            Map<String, Object> tokenMap = _jsonUtils.convertJsonToMap(providerInfo);
                           
                            if (tokenMap.containsKey("token")) 
                            {
                                // Get the token which proves that the temporary sign-up worked
                                token = (String) tokenMap.get("token");
                            }
                        }
                        break;
                    case 403: 
                        throw new IllegalStateException("The CMS back-office refused the connection");
                    default:
                        throw new IllegalStateException("The CMS back-office returned an error");
                }

                return token;
            }
        }
        catch (IOException e)
        {
            throw new RuntimeException("Unable to temporary sign the user: ", e);
        }
    }

    /**
     * Get the sign-up page of a site for a population and a language
     * @param site The site from which we are looking for the sign-up page
     * @param language The language of the wanted sign-up page
     * @param userPopulation The population of the wanted sign-up page
     * @return The parameters of the sign-up page (language, redirectPageUrl and userDirectoryID) or null if none was found
     */
    private SignupPage _getSignupPageInformations(Site site, String language, UserPopulation userPopulation) 
    {
        List<SignupPage>  signupPages = site.getSignupPages(language);
        
        if (signupPages != null && signupPages.size() >= 1)
        {
            for (SignupPage signupPage : signupPages) 
            {
                for (String populationId : signupPage.userDirByPop().keySet())
                {
                    if (StringUtils.equals(userPopulation.getId(), populationId))
                    {
                        return signupPage;
                    }
                }
            }
        }

        return null;
    }

    /**
     * Get the informations of the redirect page (the sign-up page)
     * @param request The request
     * @param site The site
     * @return The parameters of the redirect page (language, redirectPageUrl and userDirectoryID) or null if none was found
     */
    private SignupPage _getRedirectPageInformations(Request request, Site site, UserPopulation userPopulation) 
    {
        // Try to get a sign-up page of the user's preferred language
        SignupPage page = null;
        String languageByDefault = "en";
        List<String> languages = site.getLanguages();
        Locale local = request.getLocale();

        if (local != null) 
        {
            String prefLanguage = local.getLanguage();
            if (prefLanguage != null) 
            {
                if (!prefLanguage.equals(languageByDefault)) 
                {
                 // If the preferred language is a language of the site
                    if (languages.contains(prefLanguage)) 
                    {
                        // We check if the site has a sign-up page for this language
                        page = _getSignupPageInformations(site, prefLanguage, userPopulation);
                        if (page != null) 
                        {
                            return page;
                        }
                    } 
                }
            }
        }

        // If we did not find a sign-up page with the user's preferred language we try with the default language : English
        if (languages.contains(languageByDefault)) 
        {
            page = _getSignupPageInformations(site, languageByDefault, userPopulation);
            if (page != null) 
            {
                return page;
            }
        }
        // If we did not find a sign-up page with the default language,
        // We try every language of the site until we found one that has a sign-up page
        for (String language : languages)
        {
            page = _getSignupPageInformations(site, language, userPopulation);
            if (page != null)
            {
                return page;
            }
        } 

        return page;
    }

    @Override
    public void unexistingUser(String login, String firstName, String lastName, UserPopulation userPopulation, Redirector redirector, Request request) throws AccessDeniedException, ServiceUnavailableException
    {
        Site site = (Site) request.getAttribute("site");

        try
        {
            // Get the redirect page's URL, ID and language
            SignupPage signupPage = _getRedirectPageInformations(request, site, userPopulation);
            if (signupPage != null)
            {
                String redirectPageURL = signupPage.url();
                String userPopulationId = userPopulation.getId();
                String userDirectoryId = signupPage.userDirByPop().get(userPopulationId);
                String language = signupPage.lang();
                String siteName = signupPage.siteName();

                if (redirectPageURL != null) 
                {
                    // Temporary sign the user in
                    String token = _temporarySignup(siteName, userPopulationId, userDirectoryId, language, login);
                    // Redirect the user to the sign-up page
                    redirector.redirect(true, _buildRedirectURI(request, redirectPageURL, login, firstName, lastName, token));
                }
            }
        }
        catch (ProcessingException | IOException | AccessDeniedException | IllegalStateException e)
        {
            getLogger().error("The user couldn't be redirected to the sign-up page", e);
            throw new AccessDeniedException("The user couldn't be redirected to the sign-up page");
        } 
    }
}
