/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.authentication.msal;

import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.extrausermgt.users.entraid.EntraIDUserDirectory;

import com.azure.identity.ClientSecretCredential;
import com.azure.identity.ClientSecretCredentialBuilder;
import com.microsoft.aad.msal4j.IAuthenticationResult;
import com.microsoft.graph.models.User;
import com.microsoft.graph.serviceclient.GraphServiceClient;

/**
 * Sign in through Entra ID, using the OpenId Connect protocol.
 */
public class EntraIDCredentialProvider extends AbstractMSALCredentialProvider implements Serviceable
{
    private String _tenant;
    private String _loginAttribute;
    
    private OIDCScopesExtensionPoint _scopesExtensionPoint;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _scopesExtensionPoint = (OIDCScopesExtensionPoint) manager.lookup(OIDCScopesExtensionPoint.ROLE);
    }
    
    @Override
    public void init(String id, String cpModelId, Map<String, Object> paramValues, String label) throws Exception
    {
        super.init(id, cpModelId, paramValues, label);
        
        _tenant = (String) paramValues.get("authentication.entraid.tenant");

        String clientID = (String) paramValues.get("authentication.entraid.appid");
        String clientSecret = (String) paramValues.get("authentication.entraid.clientsecret");
        boolean silent = (boolean) paramValues.get("authentication.entraid.silent");
        boolean prompt = (boolean) paramValues.get("authentication.entraid.prompt");

        init(clientID, clientSecret, prompt, silent);
        
        _loginAttribute = (String) paramValues.get("authentication.entraid.loginattribute");
    }
    
    @Override
    protected String getAuthority()
    {
        return "https://login.microsoftonline.com/" + _tenant;
    }
    
    @Override
    protected Set<String> getScopes()
    {
        return _scopesExtensionPoint.getScopes();
    }
    
    @Override
    protected String getLogin(IAuthenticationResult result)
    {
        String upn = result.account().username();
        
        if (_loginAttribute.equals(EntraIDUserDirectory.ON_PREMISES_SAM_ACCOUNT_NAME))
        {
            // get the samAccountName from Entra ID
            ClientSecretCredential clientSecretCredential = new ClientSecretCredentialBuilder().clientId(_clientID)
                                                                                               .clientSecret(_clientSecret)
                                                                                               .tenantId(_tenant)
                                                                                               .build();

            GraphServiceClient graphClient = new GraphServiceClient(clientSecretCredential);
            
            User user = graphClient.users().byUserId(upn).get(requestConfiguration -> {
                requestConfiguration.queryParameters.select = new String[]{"onPremisesSamAccountName"};
            });
            
            if (user != null && StringUtils.isNotBlank(user.getOnPremisesSamAccountName()))
            {
                return user.getOnPremisesSamAccountName();
            }
        }

        return upn;
    }
}
