/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.oauth;

import java.util.Optional;

import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;

import com.nimbusds.oauth2.sdk.id.State;
import com.nimbusds.oauth2.sdk.token.AccessToken;

/**
 * Extension point registering all the OAuth provider currently available
 */
public class OauthProviderExtensionPoint extends AbstractThreadSafeComponentExtensionPoint<OAuthProvider>
{
    /** The avalon role */
    public static final String ROLE = OauthProviderExtensionPoint.class.getName();
    
    /**
     * Get the stored access token if it exist.
     * 
     * If a token exists, but is expired, this method will silently try to refresh it.
     * @param providerId the provider that granted the token
     * @return the access token or empty
     */
    public Optional<AccessToken> getStoredAccessToken(String providerId)
    {
        return getExtension(providerId).getStoredAccessToken();
    }
    
    /**
     * Get the provider linked to the state.
     * @param state the state linked to a provider
     * @return the provider id
     * @throws IllegalStateException if the state is not linked to a provider
     */
    public OAuthProvider getProviderForState(State state)
    {
        for (String id : getExtensionsIds())
        {
            if (getExtension(id).isKnownState(state))
            {
                return getExtension(id);
            }
        }
        return null;
    }
}
