/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.workspaces.extrausermgt.authentication.oidc;

import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.acting.AbstractAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Session;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.plugins.extrausermgt.authentication.oidc.AbstractOIDCCredentialProvider;

/**
 * Proxy callback for OpenId Connect protocol, so that one has to provide only one redirect URL to the identity provider.
 */
public class OIDCCallbackAction extends AbstractAction implements ThreadSafe
{
    /** Callback URL for all Ametys OIDC implementations */
    public static final String CALLBACK_URL = "/_extra-user-management/oidc-callback";
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Session session = request.getSession(true);
        
        String redirectUri = (String) session.getAttribute(AbstractOIDCCredentialProvider.REDIRECT_URI_SESSION_ATTRIBUTE);

        if (redirectUri == null)
        {
            throw new IllegalArgumentException("OIDC callback must have a redirect URI");
        }
        
        // handle errors
        String error = request.getParameter("error");
        String errorDescription = request.getParameter("error_description");
        if (error != null || errorDescription != null) 
        {
            getLogger().warn(String.format("Received an error from OpenID provider. Redirecting to initial URI. Error: %s %nErrorDescription: %s", error, errorDescription));
            redirector.redirect(true, redirectUri);
        }
        else
        {
            String queryString = request.getQueryString();
            String actualRedirectUri = redirectUri.contains("?") ? redirectUri + "&" + queryString : redirectUri + "?" + queryString;
            
            redirector.redirect(true, actualRedirectUri);
        }
        
        return EMPTY_MAP;
    }
}
