/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.authentication.oidc;

import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.Map;

import org.apache.commons.io.IOUtils;

import org.ametys.runtime.authentication.AccessDeniedException;

import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.openid.connect.sdk.OIDCScopeValue;
import com.nimbusds.openid.connect.sdk.op.OIDCProviderMetadata;


/**
 * Sign in through an OIDC application (finding URIs by itself) using the OpenId Connect protocol.
 */
public class AutoDiscoveringOIDCCredentialProvider extends AbstractOIDCCredentialProvider
{
    /** 
     * Returns the provider meta data which contains the URIs retrieved from the "/.well-known/openid-configuration" of the issuer
     * @return The provider meta data which contains the URIs
     */
    private OIDCProviderMetadata _getProviderMetadata(URI issuerURI) throws Exception
    {
        URL providerConfigurationURL = issuerURI.resolve(".well-known/openid-configuration").toURL();
        try (InputStream stream = providerConfigurationURL.openStream()) 
        {
            // Read all data from URL
            String providerInfo = IOUtils.toString(stream, StandardCharsets.UTF_8);
            return OIDCProviderMetadata.parse(providerInfo);
        }
    }
    
    @Override
    protected void initUrisScope() throws AccessDeniedException
    {  
        Map<String, Object> paramValues = getParameterValues();
        try
        {  
            URI issuerURI = URI.create((String) paramValues.get("authentication.oidc.issuerURI"));
            OIDCProviderMetadata providerMetadata = _getProviderMetadata(issuerURI);
            _authUri = providerMetadata.getAuthorizationEndpointURI();
            _tokenEndpointUri = providerMetadata.getTokenEndpointURI();
            _iss = providerMetadata.getIssuer();
            _jwkSetURL = providerMetadata.getJWKSetURI().toURL();
            _userInfoEndpoint = providerMetadata.getUserInfoEndpointURI();
            
            _scope = Scope.parse((String) paramValues.get("authentication.oidc.scopes"));
            _scope.add(OIDCScopeValue.OPENID);
        }
        catch (Exception e)
        {
            getLogger().error("Encountered a problem while retrieving provider metadata", e);
            throw new AccessDeniedException("Encountered a problem while retrieving provider metadata");
        }
    }
}
