/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.groups.entraid;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.directory.GroupDirectory;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.SizeUtils.ExcludeFromSizeCalculation;
import org.ametys.plugins.core.impl.user.directory.CachingUserAndGroupDirectoryHelper;
import org.ametys.plugins.extrausermgt.users.entraid.EntraIDUserDirectory;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import com.azure.identity.ClientSecretCredential;
import com.azure.identity.ClientSecretCredentialBuilder;
import com.microsoft.graph.core.tasks.PageIterator;
import com.microsoft.graph.models.GroupCollectionResponse;
import com.microsoft.graph.models.User;
import com.microsoft.graph.models.UserCollectionResponse;
import com.microsoft.graph.serviceclient.GraphServiceClient;

/**
 * {@link GroupDirectory} listing groups from Entra ID (Azure Active Directory).
 */
public class EntraIDGroupDirectory extends AbstractLogEnabled implements GroupDirectory, Serviceable, Disposable
{
    private static final String __PARAM_ASSOCIATED_USERDIRECTORY_ID = "org.ametys.plugins.extrausermgt.groups.entraid.userdirectory";
    private static final String __PARAM_APP_ID = "org.ametys.plugins.extrausermgt.groups.entraid.appid";
    private static final String __PARAM_CLIENT_SECRET = "org.ametys.plugins.extrausermgt.groups.entraid.clientsecret";
    private static final String __PARAM_TENANT_ID = "org.ametys.plugins.extrausermgt.groups.entraid.tenant";
    private static final String __PARAM_FILTER = "org.ametys.plugins.extrausermgt.groups.entraid.filter";
    
    private static final String[] __GROUP_ATTRIBUTES_SELECT = new String[]{"id", "displayName"};
    
    private static final String __ENTRAID_GROUPDIRECTORY_GROUP_BY_ID_CACHE_NAME_PREFIX = EntraIDGroupDirectory.class.getName() + "$group.by.id$";
    private static final String __ENTRAID_GROUPDIRECTORY_GROUPS_BY_USER_CACHE_NAME_PREFIX = EntraIDGroupDirectory.class.getName() + "$groups.by.user$";
    private static final String __ENTRAID_GROUPDIRECTORY_USERS_BY_GROUP_CACHE_NAME_PREFIX = EntraIDGroupDirectory.class.getName() + "$users.by.group$";
    private static final String __ENTRAID_GROUPDIRECTORY_ALL_GROUPS_CACHE_NAME_PREFIX = EntraIDGroupDirectory.class.getName() + "$all.groups$";
    
    private String _id;
    private I18nizableText _label;
    private String _groupDirectoryModelId;
    private Map<String, Object> _paramValues;
    private GraphServiceClient _graphClient;
    private String _filter;
    
    private String _associatedUserDirectoryId;
    private String _associatedPopulationId;
    
    private AbstractCacheManager _cacheManager;
    private CachingUserAndGroupDirectoryHelper _cacheHelper;
    private UserPopulationDAO _userPopulationDAO;
    private UserManager _userManager;
    private String _cacheGroupById;
    private String _cacheGroupsByUserId;
    private String _cacheUsersByGroupId;
    private String _cacheAllGroupsId;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _cacheManager = (AbstractCacheManager) serviceManager.lookup(AbstractCacheManager.ROLE);
        _cacheHelper = (CachingUserAndGroupDirectoryHelper) serviceManager.lookup(CachingUserAndGroupDirectoryHelper.ROLE);
        _userPopulationDAO = (UserPopulationDAO) serviceManager.lookup(UserPopulationDAO.ROLE);
    }
    
    @Override
    public String getId()
    {
        return _id;
    }

    @Override
    public I18nizableText getLabel()
    {
        return _label;
    }

    @Override
    public void setId(String id)
    {
        _id = id;
    }

    @Override
    public void setLabel(I18nizableText label)
    {
        _label = label;
    }

    @Override
    public String getGroupDirectoryModelId()
    {
        return _groupDirectoryModelId;
    }

    @Override
    public Map<String, Object> getParameterValues()
    {
        return _paramValues;
    }
    
    private void _createCaches()
    {
        Long cacheExpiration = (Long) _paramValues.get("runtime.groups.cache.expiration");
        _cacheGroupById = __ENTRAID_GROUPDIRECTORY_GROUP_BY_ID_CACHE_NAME_PREFIX + getId();
        _cacheHelper.getOrCreateCache(_cacheGroupById, _buildI18n("PLUGINS_EXTRAUSERMGT_GROUPS_ENTRA_CACHE_GROUP_BY_ID_LABEL"), _buildI18n("PLUGINS_EXTRAUSERMGT_GROUPS_ENTRA_CACHE_GROUP_BY_ID_DESC"), cacheExpiration);
        
        _cacheGroupsByUserId = __ENTRAID_GROUPDIRECTORY_GROUPS_BY_USER_CACHE_NAME_PREFIX + getId();
        _cacheHelper.getOrCreateCache(_cacheGroupsByUserId, _buildI18n("PLUGINS_EXTRAUSERMGT_GROUPS_ENTRA_CACHE_GROUPS_BY_USER_LABEL"), _buildI18n("PLUGINS_EXTRAUSERMGT_GROUPS_ENTRA_CACHE_GROUPS_BY_USER_DESC"), cacheExpiration);
        
        _cacheUsersByGroupId = __ENTRAID_GROUPDIRECTORY_USERS_BY_GROUP_CACHE_NAME_PREFIX + getId();
        _cacheHelper.getOrCreateCache(_cacheUsersByGroupId, _buildI18n("PLUGINS_EXTRAUSERMGT_GROUPS_ENTRA_CACHE_USERS_BY_GROUP_LABEL"), _buildI18n("PLUGINS_EXTRAUSERMGT_GROUPS_ENTRA_CACHE_USERS_BY_GROUP_DESC"), cacheExpiration);
        
        _cacheAllGroupsId = __ENTRAID_GROUPDIRECTORY_ALL_GROUPS_CACHE_NAME_PREFIX + getId();
        _cacheHelper.getOrCreateCache(_cacheAllGroupsId, _buildI18n("PLUGINS_EXTRAUSERMGT_GROUPS_ENTRA_CACHE_ALL_GROUPS_LABEL"), _buildI18n("PLUGINS_EXTRAUSERMGT_GROUPS_ENTRA_CACHE_ALL_GROUPS_DESC"), cacheExpiration);
    }
    
    private I18nizableText _buildI18n(String i18nKey)
    {
        String catalogue = "plugin.extra-user-management";
        I18nizableText groupDirectoryId = new I18nizableText(getId());
        Map<String, I18nizableTextParameter> labelParams = Map.of("id", groupDirectoryId);
        return new I18nizableText(catalogue, i18nKey, labelParams);
    }
    
    private Cache<String, Group> _getCacheGroupById()
    {
        return _cacheManager.get(_cacheGroupById);
    }
    
    private Cache<UserIdentity, Set<String>> _getCacheGroupsByUser()
    {
        return _cacheManager.get(_cacheGroupsByUserId);
    }
    
    private Cache<GroupIdentity, Set<UserIdentity>> _getCacheUsersByGroup()
    {
        return _cacheManager.get(_cacheUsersByGroupId);
    }
    
    private Cache<String, Set<String>> _getCacheAllGroups()
    {
        return _cacheManager.get(_cacheAllGroupsId);
    }
    
    public void init(String groupDirectoryModelId, Map<String, Object> paramValues) throws Exception
    {
        _groupDirectoryModelId = groupDirectoryModelId;
        _paramValues = paramValues;
        
        String populationAndUserDirectory = (String) paramValues.get(__PARAM_ASSOCIATED_USERDIRECTORY_ID);
        String[] split = populationAndUserDirectory.split("#");
        _associatedPopulationId = split[0];
        _associatedUserDirectoryId = split[1];
        
        String clientID = (String) paramValues.get(__PARAM_APP_ID);
        String clientSecret = (String) paramValues.get(__PARAM_CLIENT_SECRET);
        String tenant = (String) paramValues.get(__PARAM_TENANT_ID);
        _filter = (String) paramValues.get(__PARAM_FILTER);
        
        ClientSecretCredential clientSecretCredential = new ClientSecretCredentialBuilder().clientId(clientID)
                                                                                           .clientSecret(clientSecret)
                                                                                           .tenantId(tenant)
                                                                                           .build();
        
        _graphClient = new GraphServiceClient(clientSecretCredential);
        
        _createCaches();
    }

    public Group getGroup(String groupID)
    {
        return _getCacheGroupById().get(groupID, id -> {
            try
            {
                com.microsoft.graph.models.Group graphGroup = _graphClient.groups().byGroupId(groupID).get();
                return new EntraIDGroup(graphGroup.getId(), graphGroup.getDisplayName(), this, getLogger());
            }
            catch (Exception e)
            {
                getLogger().warn("Unable to retrieve group '{}' from Entra ID", groupID, e);
            }
            
            return null;
        });
    }
    
    public Set<String> getUserGroups(UserIdentity userIdentity)
    {
        return _getCacheGroupsByUser().get(userIdentity, userId -> {
            String populationId = userIdentity.getPopulationId();

            UserDirectory userDirectory = _userManager.getUserDirectory(populationId, userIdentity.getLogin());
            
            if (userDirectory == null || !populationId.equals(_associatedPopulationId) || !_associatedUserDirectoryId.equals(userDirectory.getId()))
            {
                // The user does not belong to the population or the user directory is not the Entra ID one
                return Set.of();
            }

            if (!(userDirectory instanceof EntraIDUserDirectory entraUserDirectory))
            {
                getLogger().warn("The Entra ID group directory '{}' must be associated with a Entra ID user directory.", getId());
                return Set.of();
            }
            
            Set<String> groups = new HashSet<>();
            
            try
            {
                String userIdentifier = userIdentity.getLogin();
                
                Map<String, Object> paramValues = entraUserDirectory.getParameterValues();
                String loginAttribute = (String) paramValues.get("org.ametys.plugins.extrausermgt.users.entraid.loginattribute");
                
                // If we're using SAM account names, we need to find the user first to get their UPN
                String userPrincipalNameForQuery = userIdentifier;
                if (EntraIDUserDirectory.ON_PREMISES_SAM_ACCOUNT_NAME.equals(loginAttribute))
                {
                    // Search for the user by SAM account name to get their UPN
                    try
                    {
                        List<User> users = _graphClient.users().get(requestConfiguration -> {
                            requestConfiguration.headers.add("ConsistencyLevel", "eventual");
                            requestConfiguration.queryParameters.count = true;
                            requestConfiguration.queryParameters.filter = "onPremisesSamAccountName eq '" + userIdentifier + "'";
                            requestConfiguration.queryParameters.select = new String[]{"userPrincipalName", "onPremisesSamAccountName"};
                        }).getValue();
                        
                        if (!users.isEmpty())
                        {
                            userPrincipalNameForQuery = users.get(0).getUserPrincipalName();
                        }
                        else
                        {
                            // If not found by SAM, assume the login is already a UPN (fallback case)
                            getLogger().debug("Unable to find user by SAM account name '{}', trying with UPN", userIdentifier);
                            userPrincipalNameForQuery = userIdentifier;
                        }
                    }
                    catch (Exception e)
                    {
                        getLogger().warn("Unable to find user by SAM account name '{}', trying with UPN", userIdentifier, e);
                        userPrincipalNameForQuery = userIdentifier;
                    }
                }
                
                // Get user groups using transitive membership with the UPN
                GroupCollectionResponse memberOfResponse = _graphClient.users().byUserId(userPrincipalNameForQuery).memberOf().graphGroup().get(requestConfiguration -> {
                    requestConfiguration.queryParameters.select = new String[]{"id"};
                    
                    // Filter to only get Microsoft 365 groups (Unified groups)
                    String filter = "groupTypes/any(c:c eq 'Unified')";
                    if (StringUtils.isNotEmpty(_filter))
                    {
                        filter += " and " + _filter;
                    }
                    
                    requestConfiguration.queryParameters.filter = filter;
                });
                
                // Use PageIterator to handle pagination
                new PageIterator.Builder<com.microsoft.graph.models.Group, GroupCollectionResponse>()
                                .client(_graphClient)
                                .collectionPage(memberOfResponse)
                                .collectionPageFactory(GroupCollectionResponse::createFromDiscriminatorValue)
                                .processPageItemCallback(group -> {
                                    groups.add(group.getId());
                                    return true; // Continue iteration
                                })
                                .build()
                                .iterate();
            }
            catch (Exception e)
            {
                getLogger().error("Error while fetching groups for user " + userIdentity.getLogin(), e);
                return Set.of();
            }
            
            return groups;
        });
    }

    public Set<Group> getGroups()
    {
        String cacheKey = "ALL_GROUPS"; // Cache key for all groups
        
        Set<String> groupIds = _getCacheAllGroups().get(cacheKey, key -> {
            Set<Group> groups = new HashSet<>(getGroups(-1, 0, Collections.emptyMap()));
            
            return groups.stream()
                         .map(Group::getIdentity)
                         .map(GroupIdentity::getId)
                         .collect(Collectors.toSet());
        });
        
        return groupIds.stream()
                       .map(this::getGroup)
                       .filter(Objects::nonNull)
                       .collect(Collectors.toSet());
    }

    public List<Group> getGroups(int count, int offset, Map parameters)
    {
        GroupCollectionResponse groupCollectionResponse = _graphClient.groups().get(requestConfiguration -> {
            requestConfiguration.headers.add("ConsistencyLevel", "eventual");
            
            String pattern = parameters != null ? (String) parameters.get("pattern") : null;
            
            if (StringUtils.isNotEmpty(pattern))
            {
                requestConfiguration.queryParameters.search = "\"displayName:" + pattern + "\"";
            }
            
            if (count > 0 && count < Integer.MAX_VALUE)
            {
                requestConfiguration.queryParameters.top = Math.min(count + offset, 999); // try to do only one request to Graph API
            }
            
            // List only Microsoft 365 groups
            String filter = "groupTypes/any(c:c eq 'Unified')";
            if (StringUtils.isNotEmpty(_filter))
            {
                filter += " and " + _filter;
            }
            
            requestConfiguration.queryParameters.filter = filter;
            
            requestConfiguration.queryParameters.select = __GROUP_ATTRIBUTES_SELECT;
        });
        
        List<Group> result = new ArrayList<>();
        AtomicInteger offsetCounter = new AtomicInteger(offset); // use AtomicInteger to be able to decrement directly in the below lambda
        
        try
        {
            new PageIterator.Builder<com.microsoft.graph.models.Group, GroupCollectionResponse>()
                            .client(_graphClient)
                            .collectionPage(groupCollectionResponse)
                            .collectionPageFactory(GroupCollectionResponse::createFromDiscriminatorValue)
                            .processPageItemCallback(group -> {
                                // If we have an offset, skip the first 'offset' groups
                                if (offsetCounter.decrementAndGet() <= 0)
                                {
                                    _handleGroup(group, result);
                                }
                                
                                // continue iteration if we have not reached the count limit
                                return count <= 0 || result.size() < count;
                            })
                            .build()
                            .iterate();
        }
        catch (Exception e)
        {
            getLogger().error("Error while fetching groups from Entra ID", e);
            return List.of();
        }
        
        return result;
    }
    
    private void _handleGroup(com.microsoft.graph.models.Group group, List<Group> groups)
    {
        Group storedGroup = new EntraIDGroup(group.getId(), group.getDisplayName(), this, getLogger());
        groups.add(storedGroup);
        
        // Store group in the individual cache
        _getCacheGroupById().put(group.getId(), storedGroup);
    }
    
    @Override
    public void dispose()
    {
        _cacheHelper.releaseCache(_cacheGroupById);
        _cacheHelper.releaseCache(_cacheGroupsByUserId);
        _cacheHelper.releaseCache(_cacheUsersByGroupId);
        _cacheHelper.releaseCache(_cacheAllGroupsId);
    }
    
    private static class EntraIDGroup implements Group
    {
        private String _id;
        private String _label;
        
        @ExcludeFromSizeCalculation
        private Logger _logger;
        
        @ExcludeFromSizeCalculation
        private EntraIDGroupDirectory _directory;

        public EntraIDGroup(String id, String label, EntraIDGroupDirectory directory, Logger logger)
        {
            _id = id;
            _label = label;
            _directory = directory;
            _logger = logger;
        }
        
        @Override
        public String getLabel()
        {
            return _label;
        }
        
        public GroupIdentity getIdentity()
        {
            return new GroupIdentity(_id, _directory.getId());
        }
        
        @Override
        public GroupDirectory getGroupDirectory()
        {
            return _directory;
        }

        public Set<UserIdentity> getUsers()
        {
            GroupIdentity groupIdentity = getIdentity();
            
            return _directory._getCacheUsersByGroup().get(groupIdentity, key -> {
                // if not in cache, fetch the users from the directory
                Set<UserIdentity> users = new HashSet<>();
                
                try
                {
                    UserCollectionResponse membersResponse = _directory._graphClient.groups().byGroupId(_id).members().graphUser().get(requestConfiguration -> {
                        requestConfiguration.queryParameters.select = new String[]{"userPrincipalName", "onPremisesSamAccountName"};
                    });
                    
                    // Get the associated user directory to check its login attribute configuration
                    UserDirectory associatedUserDirectory = _directory._userPopulationDAO.getUserPopulation(_directory._associatedPopulationId).getUserDirectory(_directory._associatedUserDirectoryId);
                    
                    if (!(associatedUserDirectory instanceof EntraIDUserDirectory entraUserDirectory))
                    {
                        _logger.warn("An Entra ID group directory must be associated with an Entra ID user directory.");
                        return Set.of();
                    }
                    
                    // Utiliser PageIterator pour gérer la pagination
                    new PageIterator.Builder<User, UserCollectionResponse>()
                                    .client(_directory._graphClient)
                                    .collectionPage(membersResponse)
                                    .collectionPageFactory(UserCollectionResponse::createFromDiscriminatorValue)
                                    .processPageItemCallback(user -> {
                                        users.add(new UserIdentity(entraUserDirectory.getUserIdentifier(user), _directory._associatedPopulationId));
                                        return true;
                                    })
                                    .build()
                                    .iterate();
                }
                catch (Exception e)
                {
                    _logger.error("Error while fetching members for Entra ID group " + _id, e);
                }
                
                return users;
            });
        }
        
        @Override
        public boolean equals(Object another)
        {
            if (another == null || !(another instanceof EntraIDGroup otherGroup))
            {
                return false;
            }
            
            return _id != null && _id.equals(otherGroup._id);
        }
        
        @Override
        public int hashCode()
        {
            return _id.hashCode();
        }
    }
}
