/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extrausermgt.users.entraid;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.directory.NotUniqueUserException;
import org.ametys.core.user.directory.StoredUser;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.plugins.core.impl.user.directory.AbstractCachingUserDirectory;

import com.azure.identity.ClientSecretCredential;
import com.azure.identity.ClientSecretCredentialBuilder;
import com.microsoft.graph.core.tasks.PageIterator;
import com.microsoft.graph.models.User;
import com.microsoft.graph.models.UserCollectionResponse;
import com.microsoft.graph.models.odataerrors.ODataError;
import com.microsoft.graph.serviceclient.GraphServiceClient;

/**
 * {@link UserDirectory} listing users in Entra ID.
 */
public class EntraIDUserDirectory extends AbstractCachingUserDirectory
{
    /** Constant for onPremisesSamAccountName attribute */
    public static final String ON_PREMISES_SAM_ACCOUNT_NAME = "onPremisesSamAccountName";

    private static final String[] __USER_ATTRIBUTES_SELECT = new String[]{"userPrincipalName", "surname", "givenName", "mail", "onPremisesSamAccountName"};
    
    private GraphServiceClient _graphClient;
    private String _filter;
    private String _loginAttribute;

    @Override
    public void init(String id, String udModelId, Map<String, Object> paramValues, String label) throws Exception
    {
        super.init(id, udModelId, paramValues, label);
        
        String clientID = (String) paramValues.get("org.ametys.plugins.extrausermgt.users.entraid.appid");
        String clientSecret = (String) paramValues.get("org.ametys.plugins.extrausermgt.users.entraid.clientsecret");
        String tenant = (String) paramValues.get("org.ametys.plugins.extrausermgt.users.entraid.tenant");
        _filter = (String) paramValues.get("org.ametys.plugins.extrausermgt.users.entraid.filter");
        _loginAttribute = (String) paramValues.get("org.ametys.plugins.extrausermgt.users.entraid.loginattribute");
        
        ClientSecretCredential clientSecretCredential = new ClientSecretCredentialBuilder().clientId(clientID)
                                                                                           .clientSecret(clientSecret)
                                                                                           .tenantId(tenant)
                                                                                           .build();
        
        _graphClient = new GraphServiceClient(clientSecretCredential);
    }

    @Override
    protected String getCacheTypeLabel()
    {
        return "EntraID";
    }

    public Collection<StoredUser> getStoredUsers()
    {
        return getStoredUsers(-1, 0, null);
    }

    public List<StoredUser> getStoredUsers(int count, int offset, Map<String, Object> parameters)
    {
        UserCollectionResponse userCollectionResponse = _graphClient.users().get(requestConfiguration -> {
            requestConfiguration.headers.add("ConsistencyLevel", "eventual");
            
            String pattern = parameters != null ? (String) parameters.get("pattern") : null;
            
            if (StringUtils.isNotEmpty(pattern))
            {
                requestConfiguration.queryParameters.search = "\"givenName:" + pattern + "\" OR \"surname:" + pattern + "\" OR \"userPrincipalName:" + pattern + "\"";
            }
            
            if (count > 0 && count < Integer.MAX_VALUE)
            {
                requestConfiguration.queryParameters.top = Math.min(count + offset, 999); // try to do only one request to Graph API
            }
            
            if (StringUtils.isNotEmpty(_filter))
            {
                requestConfiguration.queryParameters.filter = _filter;
                
                if (StringUtils.isEmpty(pattern))
                {
                    // if we have a filter but no pattern, we have to ask for count, to comply with MSGraph API
                    requestConfiguration.queryParameters.count = true;
                }
            }
            
            requestConfiguration.queryParameters.select = __USER_ATTRIBUTES_SELECT;
        });

        List<StoredUser> result = new ArrayList<>();
        AtomicInteger offsetCounter = new AtomicInteger(offset); // use AtomicInteger to be able to decrement directly in the below lambda
        
        try
        {
            new PageIterator.Builder<User, UserCollectionResponse>()
                            .client(_graphClient)
                            .collectionPage(userCollectionResponse)
                            .collectionPageFactory(UserCollectionResponse::createFromDiscriminatorValue)
                            .processPageItemCallback(user -> {
                                if (offsetCounter.decrementAndGet() <= 0)
                                {
                                    _handleUser(user, result);
                                }
                                
                                return count <= 0 || result.size() < count;
                            })
                            .build()
                            .iterate();
        }
        catch (Exception e)
        {
            getLogger().error("Error while fetching users from Entra ID", e);
            return Collections.emptyList();
        }
        
        return result;
    }
    
    /**
     * Get the user identifier based on the configured login attribute.
     * @param user The Azure AD user
     * @return The user identifier (either UserPrincipalName or OnPremisesSamAccountName)
     */
    public String getUserIdentifier(User user)
    {
        if (ON_PREMISES_SAM_ACCOUNT_NAME.equals(_loginAttribute))
        {
            String samAccountName = user.getOnPremisesSamAccountName();
            if (StringUtils.isNotBlank(samAccountName))
            {
                return samAccountName;
            }
            else
            {
                // Fallback to UserPrincipalName if OnPremisesSamAccountName is not available
                getLogger().debug("OnPremisesSamAccountName not available for user {}, falling back to UserPrincipalName", user.getUserPrincipalName());
                return user.getUserPrincipalName();
            }
        }
        else
        {
            // Default to UserPrincipalName
            return user.getUserPrincipalName();
        }
    }
    
    private void _handleUser(User user, List<StoredUser> storedUsers)
    {
        String userIdentifier = getUserIdentifier(user);
        StoredUser storedUser = new StoredUser(userIdentifier, user.getSurname(), user.getGivenName(), user.getMail());
        storedUsers.add(storedUser);
        
        getCacheByLogin().put(storedUser.getIdentifier(), storedUser);
    }

    public StoredUser getStoredUser(String login)
    {
        if (getCacheByLogin().hasKey(login))
        {
            StoredUser storedUser = getCacheByLogin().get(login);
            return storedUser;
        }
        
        StoredUser storedUser = null;
        try
        {
            User user = null;
            
            // Use different query strategies based on login attribute configuration
            if (ON_PREMISES_SAM_ACCOUNT_NAME.equals(_loginAttribute))
            {
                // First, try to search by onPremisesSamAccountName filter
                List<User> users = _graphClient.users().get(requestConfiguration -> {
                    requestConfiguration.headers.add("ConsistencyLevel", "eventual");
                    requestConfiguration.queryParameters.filter = "onPremisesSamAccountName eq '" + login + "'";
                    requestConfiguration.queryParameters.count = true;
                    requestConfiguration.queryParameters.select = __USER_ATTRIBUTES_SELECT;
                }).getValue();
                
                if (!users.isEmpty())
                {
                    user = users.get(0);
                }
                else
                {
                    // If not found by SAM, the login might be a UPN (fallback case)
                    // Try to search by UserPrincipalName
                    try
                    {
                        user = _graphClient.users().byUserId(login).get(requestConfiguration -> {
                            requestConfiguration.queryParameters.select = __USER_ATTRIBUTES_SELECT;
                        });
                        
                        // Verify that this user actually doesn't have a SAM account name
                        // (to ensure it's a legitimate fallback case)
                        if (user != null && StringUtils.isNotBlank(user.getOnPremisesSamAccountName()))
                        {
                            // This user has a SAM account name, so the login should have been the SAM, not the UPN
                            // This means the login provided doesn't match our configuration
                            user = null;
                        }
                    }
                    catch (Exception e)
                    {
                        // User not found by UPN either, user is null
                        getLogger().debug("User '{}' not found by SAM or UPN", login, e);
                    }
                }
            }
            else
            {
                // For UserPrincipalName, we can use the direct byUserId method
                user = _graphClient.users().byUserId(login).get(requestConfiguration -> {
                    requestConfiguration.queryParameters.select = __USER_ATTRIBUTES_SELECT;
                });
            }
            
            if (user != null)
            {
                String userIdentifier = getUserIdentifier(user);
                storedUser = new StoredUser(userIdentifier, user.getSurname(), user.getGivenName(), user.getMail());
                
                getCacheByLogin().put(storedUser.getIdentifier(), storedUser);
            }
        }
        catch (ODataError e)
        {
            // Handle ODataError specifically, which may indicate a not found error
            if (e.getResponseStatusCode() == 404)
            {
                getLogger().debug("User '{}' not found in EntraID", login);
            }
            else
            {
                getLogger().warn("Unable to retrieve user '{}' from EntraID", login, e);
            }
        }
        catch (Exception e)
        {
            getLogger().warn("Unable to retrieve user '{}' from EntraID", login, e);
        }

        return storedUser;
    }

    /*
     * As we do not know how to search for email in a "case insensitive" way, we also fill the cache "case sensitively"
     */
    public StoredUser getStoredUserByEmail(String email) throws NotUniqueUserException
    {
        if (StringUtils.isBlank(email))
        {
            return null;
        }
        
        if (getCacheByMail().hasKey(email))
        {
            StoredUser storedUser = getCacheByMail().get(email);
            return storedUser;
        }
        
        List<User> users = _graphClient.users().get(requestConfiguration -> {
            requestConfiguration.headers.add("ConsistencyLevel", "eventual");
            requestConfiguration.queryParameters.filter = "mail eq '" + email + "'";
            requestConfiguration.queryParameters.select = __USER_ATTRIBUTES_SELECT;
        }).getValue();
        
        if (users.size() == 1)
        {
            User u = users.get(0);
            String userIdentifier = getUserIdentifier(u);
            StoredUser storedUser = new StoredUser(userIdentifier, u.getSurname(), u.getGivenName(), u.getMail());
            
            getCacheByMail().put(storedUser.getEmail(), storedUser);
            
            return storedUser;
        }
        else if (users.isEmpty())
        {
            return null;
        }
        else
        {
            throw new NotUniqueUserException("Find " + users.size() + " users matching the email " + email);
        }
    }

    public CredentialsResult checkCredentials(String login, String password)
    {
        throw new UnsupportedOperationException("The EntraIDUserDirectory cannot authenticate users");
    }
}
