/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This factory creates Ametys.message.MessageTarget for a definition file of an extraction.
 * 
 * See #createTargets to know more.
 * @private
 */
Ext.define("Ametys.plugins.extraction.ExtractionDefinitionFileMessageTargetFactory",
    {
        extend: "Ametys.message.factory.DefaultMessageTargetFactory",

        /**
         * Create the targets for a message
         * @param {Object} parameters The parameters needed by the factory to create the message. Cannot be null. Handled elements are
         * @param {String} parameters.path The extraction's definition file path. Cannot be null.
         * @param {String} parameters.name The extraction's definition file name. Cannot be null.
         * @param {String} [parameters.description] The extraction's description
         * @param {String} [parameters.descriptionId] The identifier of the extraction's description
         * @param {Object} parameters.author The extraction'author.
         * @param {String} parameters.author.login The login of the extraction's author
         * @param {String} parameters.author.populationId The population of the extraction's author
         * @param {String} parameters.canRead true if the current user can read the extraction
         * @param {String} parameters.canWrite true if the current user can write the extraction
         * @param {String} parameters.canAssignRights true if the current user can edit rights of the extraction definition
         * @param {String} [parameters.nodeTag] The tag of the selected extraction's node
         * @param {Function} callback The callback function called when the targets are created. Parameters are
         * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
         */
        createTargets: function(parameters, callback)
        {
            if (!Ext.isEmpty(parameters.description))
            {
                this._getContentCb(parameters.description, parameters, callback);
            }
            else if (!Ext.isEmpty(parameters.descriptionId))
            {
                Ametys.cms.content.ContentDAO.getContent(parameters.descriptionId, Ext.bind(this._getContentCb, this, [parameters, callback], 1), null, false);
            }
            else
            {
                this._createTargetsCb([], parameters, callback);
            }
        },
        
        /**
         * Callback called when the description infos are retrieved
         * @param {Ametys.cms.content.Content} description the retrieved description
         * @param {Object} parameters The parameters needed by the factory to create the message. Cannot be null. Handled elements are
         * @param {String} parameters.path The extraction's definition file path. Cannot be null.
         * @param {String} parameters.name The extraction's definition file name. Cannot be null.
         * @param {String} [parameters.nodeTag] the tag of the selected extraction's node
         * @param {Function} callback The callback function called when the targets are created. Parameters are
         * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
         */
        _getContentCb: function(description, parameters, callback)
        {
            if (description)
            {
                var contentTargetConfig = {
                    id: Ametys.message.MessageTarget.CONTENT,
                    parameters: {
                        contents: [description]
                    }
                }
                    
                Ametys.message.MessageTargetFactory.createTargets(contentTargetConfig, Ext.bind(this._createTargetsCb, this, [parameters, callback], 1));
            }
            else
            {
                this._createTargetsCb([], parameters, callback);
            }
        },
        
        /**
         * Callback called when the content targets are created
         * @param {Ametys.message.MessageTarget[]} contentTargets the content targets for description
         * @param {Object} parameters The parameters needed by the factory to create the message. Cannot be null. Handled elements are
         * @param {String} parameters.path The extraction's definition file path. Cannot be null.
         * @param {String} parameters.name The extraction's definition file name. Cannot be null.
         * @param {String} [parameters.nodeTag] the tag of the selected extraction's node
         * @param {Function} callback The callback function called when the targets are created. Parameters are
         * @param {Ametys.message.MessageTarget[]} callback.targets The targets created. Cannot be null.
         */
        _createTargetsCb: function(contentTargets, parameters, callback)
        {
            var subtargets = contentTargets;
            
            if (!Ext.isEmpty(parameters.nodeTag))
            {
                subtargets.push(Ext.create("Ametys.message.MessageTarget", {
                    id: Ametys.message.MessageTarget.EXTRACTION_NODE,
                    parameters: {
                        tag: parameters.nodeTag
                    }
                }));
            }
            
            var target = Ext.create("Ametys.message.MessageTarget", {
                id: Ametys.message.MessageTarget.EXTRACTION_DEFINITION_FILE,
                parameters: {
                    path: parameters.path,
                    name: parameters.name,
                    author: parameters.author,
                    canRead: parameters.canRead,
                    canWrite: parameters.canWrite,
                    canDelete: parameters.canDelete,
                    canAssignRights: parameters.canAssignRights
                },
                subtargets: subtargets
            });    
                
            callback([target]);
        }
    }
);