/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.component;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContext;
import org.ametys.plugins.extraction.execution.ExtractionExecutionContextHierarchyElement;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.thesaurus.Thesaurus;
import org.ametys.plugins.thesaurus.ThesaurusDAO;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * This class represents a thesaurus component of the extraction module
 */
public class ThesaurusExtractionComponent extends AbstractExtractionComponent
{
    private String _microThesaurusId;
    private Thesaurus _thesaurus;
    private ContentType _microThesaurus;
    private int _maxLevel;
    
    private I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public void configure(Configuration thesaurus) throws ConfigurationException
    {
        super.configure(thesaurus);
        _microThesaurusId = thesaurus.getAttribute("microThesaurus");
        _maxLevel = thesaurus.getAttributeAsInteger("max-level", -1);
        
        Set<String> ancestors = _contentTypesHelper.getAncestors(_microThesaurusId);
        if (!ancestors.contains(ThesaurusDAO.MICROTHESAURUS_ABSTRACT_CONTENT_TYPE))
        {
            throw new IllegalArgumentException(getLogsPrefix() + "'" + _microThesaurusId + "' is not a valid microthesaurus id");
        }
        
        _initializeThesaurus();
    }

    void _initializeThesaurus()
    {
        _microThesaurus = _contentTypeExtensionPoint.getExtension(_microThesaurusId);
        _thesaurus = _thesaurusDAO.getParentThesaurus(_microThesaurus);
        if (_thesaurus == null)
        {
            throw new IllegalArgumentException(getLogsPrefix() + "The microthesaurus '" + _microThesaurusId + "' has no attached thesaurus.");
        }
    }

    @Override
    public void executeComponent(ContentHandler contentHandler, ExtractionExecutionContext context) throws Exception
    {
        I18nizableText i18nLabel = _microThesaurus.getLabel();
        String label = _i18nUtils.translate(i18nLabel, null); // FIXME Use user preference language
        
        AttributesImpl thesaurusAttributes = new AttributesImpl();
        thesaurusAttributes.addCDATAAttribute("thesaurusLabel", _thesaurus.getLabel());
        thesaurusAttributes.addCDATAAttribute("microThesaurusLabel", label);
        thesaurusAttributes.addCDATAAttribute("microThesaurusId", _microThesaurusId);
        XMLUtils.startElement(contentHandler, _tagName, thesaurusAttributes);
        
        AmetysObjectIterable<Content> children = _thesaurusDAO.getRootTerms(_microThesaurusId);
        _processChildren(contentHandler, children, 1, context);
        
        XMLUtils.endElement(contentHandler, _tagName);
    }
    
    public Set<String> getContentTypes()
    {
        return new HashSet<>(Collections.singletonList(_microThesaurusId));
    }

    private void _processChildren(ContentHandler contentHandler, AmetysObjectIterable<Content> children, int currentLevel, ExtractionExecutionContext context) throws Exception
    {
        long nbTermChildren = -1;
        int currentChildIndex = 0;
        if (getLogger().isDebugEnabled())
        {
            nbTermChildren = children.getSize();
        }
        
        for (Content term : children)
        {
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug(getLogsPrefix() + "executing term " + ++currentChildIndex + "/" + nbTermChildren + " at level " + currentLevel);
            }
            
            AttributesImpl attributes = new AttributesImpl();
            attributes.addCDATAAttribute("name", term.getTitle());
            XMLUtils.startElement(contentHandler, "term", attributes);

            if (_maxLevel < 0 || currentLevel < _maxLevel)
            {
                ExtractionExecutionContextHierarchyElement currentContext = new ExtractionExecutionContextHierarchyElement(this, Collections.singleton(term), false);
                executeSubComponents(contentHandler, context, currentContext);
                _processChildren(contentHandler, _thesaurusDAO.getChildTerms(_microThesaurusId, term.getId()), currentLevel + 1, context);
            }
            else
            {
                ExtractionExecutionContextHierarchyElement currentContext = new ExtractionExecutionContextHierarchyElement(this, Collections.singleton(term));
                executeSubComponents(contentHandler, context, currentContext);
            }
            
            XMLUtils.endElement(contentHandler, "term");
        }
    }
    
    @Override
    public Map<String, Object> getComponentDetailsForTree()
    {
        Map<String, Object> details = super.getComponentDetailsForTree();
        details.put("tag", ExtractionConstants.THESAURUS_COMPONENT_TAG);
        
        @SuppressWarnings("unchecked")
        Map<String, Object> data = (Map<String, Object>) details.get("data");
        data.put("thesaurusId", _thesaurus.getId());
        data.put("microThesaurusId", getMicroThesaurusId());
        data.put("maxLevel", this.getMaxLevel());
        details.put("iconCls", "ametysicon-books");
        
        return details;
    }

    @Override
    protected String getDefaultTagName()
    {
        return "thesaurus";
    }

    @Override
    protected String getLogsPrefix()
    {
        return "Thesaurus component '" + _tagName + "': ";
    }

    /**
     * Retrieves the component microthesaurus id
     * @return The microthesaurus id
     */
    public String getMicroThesaurusId()
    {
        return _microThesaurusId;
    }

    /**
     * Set the microthesaurus id
     * @param microThesaurusId The microthesaurus id to set
     */
    public void setMicroThesaurusId(String microThesaurusId)
    {
        _microThesaurusId = microThesaurusId;
    }

    /**
     * Retrieves the maximum level of specific terms to look at
     * @return The maximum level
     */
    public int getMaxLevel()
    {
        return _maxLevel;
    }

    /**
     * Set the maximum level of specific terms to look at
     * @param maxLevel The maximum level to set
     */
    public void setMaxLevel(int maxLevel)
    {
        _maxLevel = maxLevel;
    }
}
