/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.edition;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;

import org.ametys.core.file.FileHelper;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.ExtractionDAO;
import org.ametys.plugins.extraction.rights.ExtractionAccessController;

/**
 * Component for operations on extraction folders
 */
public class FoldersClientSideElement extends StaticClientSideElement
{
    private FileHelper _fileHelper;
    private SourceResolver _srcResolver;
    private ExtractionDAO _extractionDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _fileHelper = (FileHelper) serviceManager.lookup(FileHelper.ROLE);
        _srcResolver = (org.apache.excalibur.source.SourceResolver) serviceManager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);        
        _extractionDAO = (ExtractionDAO) serviceManager.lookup(ExtractionDAO.ROLE);
    }

    /**
     * Add a new folder
     * @param parentRelPath the relative parent file's path from parameters files root directory
     * @param name The name of folder to create
     * @return a map containing the name of the created folder, its path and the path of its parent
     * @throws IOException If an error occurred while creating folder
     */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> addFolder(String parentRelPath, String name) throws IOException 
    {
        String nonNullParentRelPath = StringUtils.defaultString(parentRelPath);
        
        FileSource rootDir = (FileSource) _srcResolver.resolveURI(ExtractionConstants.DEFINITIONS_DIR);
        if (!rootDir.exists())
        {
            rootDir.getFile().mkdirs();
        }
        
        String parentURI = ExtractionConstants.DEFINITIONS_DIR + (StringUtils.isNotEmpty(nonNullParentRelPath) ? "/" + nonNullParentRelPath : "");
        
        Map<String, Object> result = _fileHelper.addFolder(parentURI, name, true);
        
        if (result.containsKey("uri"))
        {
            String folderUri = (String) result.get("uri");
            // Get only the part after the root folder to get the relative path 
            String path = StringUtils.substringAfter(folderUri, rootDir.getURI());
            result.put("path", ExtractionDAO.trimLastFileSeparator(path));
            result.put("parentPath", ExtractionDAO.trimLastFileSeparator(nonNullParentRelPath));
        }

        return result;
    }
    
    /**
    * Remove a folder or a file
    * @param relPath the relative file's path from parameters files root directory
    * @return the result map
    * @throws IOException If an error occurs while removing the folder
    */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> deleteFile(String relPath) throws IOException
    {
        String fileUri = ExtractionConstants.DEFINITIONS_DIR + (relPath.length() > 0 ? "/" + relPath : "");
        String context = ExtractionAccessController.ROOT_CONTEXT + "/" + relPath;
        FileSource folderToDelete = (FileSource) _srcResolver.resolveURI(fileUri);
        _extractionDAO.deleteRightsRecursively(context, folderToDelete);
        return _fileHelper.deleteFile(fileUri);
    }
    
    /**
    * Rename a file or a folder 
    * @param relPath the relative file's path from parameters files root directory
    * @param name the new name of the file/folder
    * @return the result map
    * @throws IOException if an error occurs while renaming the file/folder
    */
    @Callable (rights = ExtractionConstants.MODIFY_EXTRACTION_RIGHT_ID)
    public Map<String, Object> renameFile(String relPath, String name) throws IOException 
    {
        String fileUri = ExtractionConstants.DEFINITIONS_DIR + relPath;
        FileSource file = (FileSource) _srcResolver.resolveURI(fileUri);
        String relativeParentPath = StringUtils.removeEnd(relPath, file.getName());
        String relativeNewFilePath = relativeParentPath + name;
        return _extractionDAO.moveOrRenameExtractionDefinitionFile(relPath, relativeNewFilePath);
    }
}
