/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.io.File;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.Extraction.ClausesVariable;
import org.ametys.plugins.extraction.execution.Extraction.ClausesVariableType;
import org.ametys.plugins.extraction.execution.pipeline.PipelineManager;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * This client site element creates a button to execute an extraction
 */
public class ExecuteExtractionClientSideElement extends StaticClientSideElement
{
    private UserManager _userManager;
    private ExtractionDefinitionReader _reader;
    private SourceResolver _sourceResolver;
    private PipelineManager _pipelineManager;
    private ContentTypeExtensionPoint _contentTypeExtensionPoint;
    private ContentTypesHelper _contentTypesHelper;
    private ExecuteExtractionRunnableHelper _executeExtractionRunnableHelper;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _reader = (ExtractionDefinitionReader) serviceManager.lookup(ExtractionDefinitionReader.ROLE);
        _sourceResolver = (SourceResolver) serviceManager.lookup(SourceResolver.ROLE);
        _pipelineManager = (PipelineManager) serviceManager.lookup(PipelineManager.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _contentTypesHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _executeExtractionRunnableHelper = (ExecuteExtractionRunnableHelper) serviceManager.lookup(ExecuteExtractionRunnableHelper.ROLE);
    }
    
    /**
     * Retrieve needed extraction parameters.
     * @param definitionFile The extraction definition file path
     * @return a <code>Map</code> containing parameters infos used to configure form to fill the parameters
     * @throws Exception if an error occurs
     */
    @Callable (rights = ExtractionConstants.EXECUTE_EXTRACTION_RIGHT_ID)
    public Map<String, Object> getExecutionParameters(String definitionFile) throws Exception
    {
        Map<String, Object> executionParameters = new LinkedHashMap<>();

        String definitionFilePath = ExtractionConstants.DEFINITIONS_DIR + definitionFile;
        Source src = _sourceResolver.resolveURI(definitionFilePath);
        File file = ((FileSource) src).getFile();
        
        if (!file.exists())
        {
            throw new IllegalArgumentException("The file " + definitionFilePath + " does not exist.");
        }

        executionParameters.put("pipeline", _getPipelineInputConfig(definitionFile));
        
        Extraction extraction = _reader.readExtractionDefinitionFile(file);
        
        List<ClausesVariable> clausesVariables = extraction.getClausesVariables();
        List<String> optionalColumns = extraction.getDisplayOptionalColumnsNames();

        if (!clausesVariables.isEmpty())
        {
            Map<String, Object> clausesVariablesFieldSet = new HashMap<>();
            clausesVariablesFieldSet.put("role", "fieldset");
            clausesVariablesFieldSet.put("label", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_CLAUSES_VARIABLES_FIELDSET_LABEL"));
        
            Map<String, Object> clausesVariablesFieldSetElements = new HashMap<>();
            for (ClausesVariable clausesVariable : clausesVariables)
            {
                clausesVariablesFieldSetElements.put(clausesVariable.name(), _getClausesVariableInputConfig(clausesVariable));
            }
            clausesVariablesFieldSet.put("elements", clausesVariablesFieldSetElements);
            
            executionParameters.put("clausesVariables", clausesVariablesFieldSet);
        }
        
        if (!optionalColumns.isEmpty())
        {
            Map<String, Object> optionalColumnsFieldSet = new HashMap<>();
            optionalColumnsFieldSet.put("role", "fieldset");
            optionalColumnsFieldSet.put("label", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_OPTIONAL_COLUMNS_FIELDSET_LABEL"));
        
            Map<String, Object> optionalColumnsFieldSetElements = new HashMap<>();
            for (String optionalColumn : optionalColumns)
            {
                optionalColumnsFieldSetElements.put(optionalColumn, _getOptionalColumnsInputConfig(optionalColumn));
            }
            optionalColumnsFieldSet.put("elements", optionalColumnsFieldSetElements);
            
            executionParameters.put("optionalColumns", optionalColumnsFieldSet);
        }
            
        executionParameters.put("recipient", _getRecipientInputConfig());
        
        return executionParameters;
    }
    
    private Map<String, Object> _getPipelineInputConfig(String definitionFile)
    {
        Map<String, Object> inputConfig = new HashMap<>();
        
        inputConfig.put("label", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_PIPELINE_INPUT_LABEL"));
        inputConfig.put("description", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_PIPELINE_INPUT_DESCRIPTION"));
        inputConfig.put("type", "string");
        
        inputConfig.put("default-value", _pipelineManager.getDefaultPipeline());
        inputConfig.put("validation", _getMandatoryValidation());
        
        inputConfig.put("widget", "edition.select-pipeline");
        
        Map<String, Object> widgetParams = new HashMap<>();
        widgetParams.put("extraction", definitionFile);
        inputConfig.put("widget-params", widgetParams);
        
        return inputConfig;
    }
    
    private Map<String, Object> _getClausesVariableInputConfig(ClausesVariable clausesVariable)
    {
        Map<String, Object> inputConfig = new HashMap<>();
        
        inputConfig.put("label", clausesVariable.name());
        
        if (ClausesVariableType.SOLR_REQUEST.equals(clausesVariable.type()))
        {
            inputConfig.put("description", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_CLAUSES_VARIABLE_SOLR_REQUEST_INPUT_DESCRIPTION"));

            inputConfig.put("type", "string");
            inputConfig.put("widget", "edition.solr-code");
            
            Map<String, Object> widgetParams = new HashMap<>();
            
            Set<String> commonAncestors = _contentTypesHelper.getCommonAncestors(clausesVariable.contentTypeIds());
            widgetParams.put("ctypes", commonAncestors);            
            widgetParams.put("singleLine", true);
            widgetParams.put("height", 66);
            widgetParams.put("mode", "text/x-solr-ametys");
            
            inputConfig.put("widget-params", widgetParams);
        }
        else
        {
            inputConfig.put("description", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_CLAUSES_VARIABLE_SELECT_CONTENTS_INPUT_DESCRIPTION"));
    
            inputConfig.put("type", "content");
            inputConfig.put("multiple", true);
            
            String widget = "edition.select-content";
            Map<String, Object> widgetParams = new HashMap<>();
            
            Optional<String> contentTypeId = clausesVariable.contentTypeIds().stream().findFirst();
            if (contentTypeId.isPresent())
            {
                ContentType contentType = _contentTypeExtensionPoint.getExtension(contentTypeId.get());
                if (contentType.isReferenceTable())
                {
                    widget = "edition.select-referencetable-content";
                }
                
                widgetParams.put("contentType", contentTypeId.get());
            }
            
            clausesVariable.searchModelId()
                           .ifPresent(searchModelId -> widgetParams.put("modelId", searchModelId));
            
            clausesVariable.solrRequest()
                           .ifPresent(solrRequest -> widgetParams.put("solrRequest", solrRequest));

            inputConfig.put("widget", widget);
            inputConfig.put("widget-params", widgetParams);
        }
        
        inputConfig.put("validation", _getMandatoryValidation());
        
        return inputConfig;
    }
    
    private Map<String, Object> _getOptionalColumnsInputConfig(String optionalColumn)
    {
        Map<String, Object> inputConfig = new HashMap<>();
        
        inputConfig.put("label", optionalColumn);
        inputConfig.put("description", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_OPTIONAL_COLUMNS_INPUTS_DESCRIPTION"));
        
        inputConfig.put("type", "boolean");
        inputConfig.put("widget", "edition.checkbox");
        
        inputConfig.put("validation", _getMandatoryValidation());
        
        return inputConfig;
    }
    
    private Map<String, Object> _getRecipientInputConfig()
    {
        Map<String, Object> inputConfig = new HashMap<>();
        
        // Get current user email
        String currentUserEmail = null;
        UserIdentity currentUser = _currentUserProvider.getUser();
        
        String login = currentUser.getLogin();
        if (StringUtils.isNotBlank(login))
        {
            String userPopulationId = currentUser.getPopulationId();
            User user = _userManager.getUser(userPopulationId, login);
            currentUserEmail = user.getEmail();
        }
        
        inputConfig.put("label", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_RECIPIENT_INPUT_LABEL"));
        inputConfig.put("description", new I18nizableText(ExtractionConstants.PLUGIN_NAME, "PLUGINS_EXTRACTION_EXECUTE_EXTRACTION_RECIPIENT_INPUT_DESCRIPTION"));
        inputConfig.put("type", "string");
        inputConfig.put("default-value", currentUserEmail);
        
        return inputConfig;
    }
    
    private Map<String, Object> _getMandatoryValidation()
    {
        Map<String, Object> mandatoryValidation = new HashMap<>();
        mandatoryValidation.put("mandatory", true);
        return mandatoryValidation;
    }
    
    /**
     * Execute the extraction
     * @param definitionFilePath The extraction definition file path
     * @param variables clauses variables and optional columns
     * @param recipient An email will be sent at this address when the extraction is complete
     * @param pipelineId The id of the extraction pipeline
     * @return a Map with error if one occurs
     * @throws Exception if an error occurs
     */
    @Callable (rights = ExtractionConstants.EXECUTE_EXTRACTION_RIGHT_ID)
    public Map<String, Object> executeExtraction(String definitionFilePath, Map<String, Object> variables, String recipient, String pipelineId) throws Exception
    {
        return _executeExtractionRunnableHelper.executeExtraction(definitionFilePath, variables, recipient, pipelineId);
    }
}
