/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.EnumUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceException;
import org.apache.excalibur.source.TraversableSource;
import org.apache.excalibur.source.impl.FileSource;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.ui.parameter.files.GetParameterFileAction;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.plugins.extraction.execution.Extraction.ExtractionProfile;

/**
 * Action for getting extraction definition files
 */
public class GetExtractionDefinitionFilesAction extends GetParameterFileAction
{
    private CurrentUserProvider _currentUserProvider;
    private ExtractionDefinitionReader _definitionReader;
    private ExtractionDAO _extractionDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _definitionReader = (ExtractionDefinitionReader) serviceManager.lookup(ExtractionDefinitionReader.ROLE);
        _extractionDAO = (ExtractionDAO) serviceManager.lookup(ExtractionDAO.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);

        String profileId = StringUtils.defaultIfEmpty((String) jsParameters.get("profile"), ExtractionProfile.READ_ACCESS.name());
        
        if (!EnumUtils.isValidEnum(ExtractionProfile.class, profileId.toUpperCase()))
        {
            throw new IllegalArgumentException("Unexpected profile identifier : " + profileId);
        }
        
        ExtractionProfile profile = ExtractionProfile.valueOf(profileId.toUpperCase());
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String path = Objects.toString(request.getParameter("path"), StringUtils.EMPTY);
        
        String rootURI = getRootURI(request);
        TraversableSource rootDir = (TraversableSource) _srcResolver.resolveURI(rootURI);
        TraversableSource currentDir = (TraversableSource) _srcResolver.resolveURI(rootURI + (path.length() > 0 ? "/" + path : ""));
        List<Map<String, Object>> nodes = _fillNodes(currentDir, profile, rootDir, userIdentity);
        
        Map<String, Object> result = new HashMap<>();
        result.put("nodes", nodes);
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
        
    private List<Map<String, Object>> _fillNodes(TraversableSource currentDir, ExtractionProfile profile, TraversableSource rootDir, UserIdentity userIdentity) throws SourceException, Exception
    {
        List<Map<String, Object>> nodes = new ArrayList<>();
        
        if (currentDir.exists())
        {
            for (TraversableSource child : (Collection<TraversableSource>) currentDir.getChildren())
            {
                if (!isIgnoredSource(child))
                {
                    if (child.isCollection())
                    {
                        Map<String, Object> properties = _extractionContainer2JsonObject(child, rootDir);

                        switch (profile)
                        {
                            case WRITE_ACCESS:
                                if ((boolean) properties.get("canWrite") || _extractionDAO.hasAnyWritableDescendant(userIdentity, child, true))
                                {
                                    nodes.add(properties);
                                }
                                break;
                            case RIGHT_ACCESS:
                                if ((boolean) properties.get("canAssignRights") || _extractionDAO.hasAnyAssignableDescendant(userIdentity, child))
                                {
                                    nodes.add(properties);
                                }
                                break;
                            case READ_ACCESS:
                            default:
                                if ((boolean) properties.get("canRead") || _extractionDAO.hasAnyReadableDescendant(userIdentity, child))
                                {
                                    nodes.add(properties);
                                }
                                break;
                        }
                    }
                    else
                    {
                        // To parse the extraction definition file only once, check here if the current user can read it
                        Extraction extraction = _definitionReader.readExtractionDefinitionFile(((FileSource) child).getFile());
                        switch (profile)
                        {
                            case WRITE_ACCESS:
                                if (_extractionDAO.canWrite(userIdentity, child))
                                {
                                    nodes.add(_extraction2JsonObject(extraction, child, rootDir));
                                }
                                break;
                            case RIGHT_ACCESS:
                                if (_extractionDAO.canAssignRights(userIdentity, child))
                                {
                                    nodes.add(_extraction2JsonObject(extraction, child, rootDir));
                                }
                                break;
                            case READ_ACCESS:
                            default:
                                if (_extractionDAO.canRead(userIdentity, child))
                                {
                                    nodes.add(_extraction2JsonObject(extraction, child, rootDir));
                                }
                                break;
                        }
                        
                    }
                }
            }
        }
        return nodes;
    }
    @Override
    protected String getRootURI(Request request)
    {
        return ExtractionConstants.DEFINITIONS_DIR;
    }

    
    @Override
    protected boolean isIgnoredSource(TraversableSource source)
    {
        return !source.isCollection() && !source.getName().endsWith(".xml");
    }
    
    private Map<String, Object> _extraction2JsonObject(Extraction extraction, TraversableSource file, TraversableSource root)
    {
        Map<String, Object> jsonObject = super._resource2JsonObject(file, root);

        jsonObject.putAll(_extractionDAO.getExtractionProperties(extraction, file));
        
        return jsonObject;
    }
    
    private Map<String, Object> _extractionContainer2JsonObject(TraversableSource file, TraversableSource root)
    {
        Map<String, Object> jsonObject = super._collection2JsonObject(file, root);

        jsonObject.putAll(_extractionDAO.getExtractionContainerProperties(file));
        
        return jsonObject;
    }

}
