/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.io.File;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.excalibur.source.TraversableSource;

import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.core.ui.parameter.files.AbstractGetFilesAction;
import org.ametys.plugins.extraction.ExtractionConstants;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * Action for getting extraction result files
 */
public class GetExtractionResultFilesAction extends AbstractGetFilesAction
{
    private RightManager _rightManager;
    private CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        if (_rightManager.hasRight(_currentUserProvider.getUser(), ExtractionConstants.EXECUTE_EXTRACTION_RIGHT_ID, "/${WorkspaceName}") != RightResult.RIGHT_ALLOW)
        {
            String errorMessage = "User " + _currentUserProvider.getUser() + " try to list extraction result files with no sufficient rights"; 
            getLogger().error(errorMessage);
            throw new IllegalStateException(errorMessage);
        }
        
        return super.act(redirector, resolver, objectModel, source, parameters);
    }
    
    @Override
    protected String getRootURI(Request request)
    {
        return new File(AmetysHomeHelper.getAmetysHomeData(), ExtractionConstants.RESULT_EXTRACTION_DIR_NAME).toURI().toString();
    }
    
    @Override
    protected Map<String, Object> _resource2JsonObject(TraversableSource file, TraversableSource root)
    {
        Map<String, Object> jsonObject = super._resource2JsonObject(file, root);
        
        String path = (String) jsonObject.get("path");
        jsonObject.put("downloadUrl", URIUtils.encodePath(path));
        
        return jsonObject;
    }
}
