/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.FilenameUtils;
import org.ametys.plugins.extraction.component.AbstractSolrExtractionComponent;
import org.ametys.plugins.extraction.component.ExtractionComponent;
import org.ametys.plugins.extraction.edition.EditExtractionNodeManager;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The resolver for string paths which can contain variables (format is <code>foo/a_${meta1/meta2/meta3}_m_${meta4}_z/bar\qux/${meta5}</code>)
 * and need to be resolved against some given contents.
 */
public class PathResolver extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role. */
    public static final String ROLE = PathResolver.class.getName();
    
    private static final List<Character> __PATH_SEPARATORS = Arrays.asList('/', '\\');
    private static final Pattern __VARIABLE_REGEXP_PATTERN = Pattern.compile(
            "\\$" // character '$' literally
            + "\\{" // character '{' literally
            + "([\\w-\\/]*)" // capturing group: [any word character or '-' or '/'] between zero and unlimited times
            + "\\}" // character '}' literally
    );
    private static final String __NO_VALUE_OR_BLANK_FOLDER_NAME = "_NOVALUE_";
    
    private EditExtractionNodeManager _editExtractionNodeManager;
    private ContentTypeExtensionPoint _contentTypeExtensionPoint;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _editExtractionNodeManager = (EditExtractionNodeManager) manager.lookup(EditExtractionNodeManager.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    /**
     * Returns <code>true</code> if the path contains variables to be resolved.
     * <br>If it returns <code>false</code>, then {@link #resolvePath(String, List, Extraction, Path)}
     * can be called with <code>null</code> parameters for contents and extraction.
     * @param path The relative path to resolve
     * @return <code>true</code> if the path contains variables to be resolved
     */
    public boolean hasVariable(String path)
    {
        Matcher m = __VARIABLE_REGEXP_PATTERN.matcher(path);
        return m.find();
    }
    
    /**
     * Returns <code>true</code> if the unresolved path represents a folder, i.e. its last element does not contain a '.' character.
     * @param path The relative path to resolve
     * @return <code>true</code> if the unresolved path represents a folder
     */
    public boolean isFolder(String path)
    {
        PathWrapper unresolvedPath = _splitPathElements(path);
        if (path.isEmpty())
        {
            return true;
        }
        
        List<String> elements = unresolvedPath.getElements();
        String lastElement = elements.get(elements.size() - 1);
        // dummy variable replacement to avoid to take account of '.' 
        // in variable names (not possible for the moment but it could change)
        Matcher m = __VARIABLE_REGEXP_PATTERN.matcher(lastElement);
        StringBuffer sb = new StringBuffer();
        while (m.find())
        {
            m.group(1);
            m.appendReplacement(sb, "");
        }
        m.appendTail(sb);
        return !sb.toString().contains(".");
    }
    
    /**
     * Resolve the given path, which can contain variables, with the values for the given contents.
     * <br>Thus, the result is a {@link Map} of resolved {@link Path Paths}, each value containg the list of contents for its associated resolved path key.
     * <br>If a variable is multivalued, a content can be in several paths at the same time in the result.
     * <br>
     * <br>For instance, <code>foo/a_${meta1/meta2/meta3}_m_${meta4}_z/bar\qux/${meta5}</code>
     * could be resolved to the path <code>foo / a_val1_m_val2_z / bar / qux / val3</code> for some contents.
     * @param path The relative path to resolve. It must not start, nor end with a '/' or a '\' character
     * @param contents The contents. Can be null if {@link PathResolver#hasVariable(String)} was called before and returned false.
     * @param extraction The extraction. Can be null if {@link PathResolver#hasVariable(String)} was called before and returned false.
     * @param basePath The base absolute path
     * @return The absolute resolved paths mapped with their matching contents.
     * <br>If the returned map contains only one path with a null list, it means that all contents match for that given single path.
     * @throws IllegalArgumentException If the path contains variables that are not in the extracted contents' model
     */
    public Map<Path, List<Content>> resolvePath(String path, List<Content> contents, Extraction extraction, Path basePath) throws IllegalArgumentException
    {
        PathWrapper unresolvedPath = _splitPathElements(path);
        Collection<ContentType> contentTypes = _getFirstLevelContentTypes(extraction);
        
        Set<String> variableNames = new HashSet<>();
        for (String element : unresolvedPath.getElements())
        {
            _fillVariableNames(contentTypes, element, variableNames);
        }
        
        if (variableNames.isEmpty())
        {
            return Collections.singletonMap(_toPath(_validPath(unresolvedPath), basePath), null);
        }
        
        Map<Content, Set<PathWrapper>> pathByContent = _pathByContent(unresolvedPath, Optional.ofNullable(contents).orElse(Collections.emptyList()));
        Map<PathWrapper, List<Content>> contentsByPath = _contentsByPath(pathByContent);
        
        return contentsByPath.entrySet()
                .stream()
                .collect(Collectors.toMap(
                    e -> _toPath(e.getKey(), basePath), 
                    e -> e.getValue()
                ));
    }
    
    private PathWrapper _validPath(PathWrapper pathWithNoVar)
    {
        List<String> pathElements = pathWithNoVar.getElements();
        if (pathElements.size() == 1 && "".equals(pathElements.get(0)))
        {
            return pathWithNoVar;
        }
        return new PathWrapper(
                pathElements.stream()
                    .map(this::_validPathElementName)
                    .collect(Collectors.toList()));
    }
    
    private Path _toPath(PathWrapper resolvedPath, Path basePath)
    {
        List<String> elements = resolvedPath.getElements();
        return Paths.get(basePath.toString(), elements.toArray(new String[elements.size()]));
    }
    
    /*
     * In:
     *      "foo/a_${meta1/meta2/meta3}_m_${meta4}_z/bar\qux/${meta5}"
     * Out:
     *      ["foo", "a_${meta1/meta2/meta3}_m_${meta4}_z", "bar", "qux", "${meta5}"]
     */
    private PathWrapper _splitPathElements(String path)
    {
        List<String> res = new ArrayList<>();
        boolean previousCharWasDollar = false;
        boolean inVariable = false;
        int start = 0;
        int end = 0;
        
        for (int i = 0; i < path.length(); i++)
        {
            char currentChar = path.charAt(i);
            if (!inVariable && __PATH_SEPARATORS.contains(currentChar))
            {
                end = i;
                res.add(path.substring(start, end));
                start = i + 1;
            }
            else if (!inVariable && currentChar == '$')
            {
                previousCharWasDollar = true;
            }
            else if (!inVariable && previousCharWasDollar && currentChar == '{')
            {
                inVariable = true;
            }
            else if (inVariable && currentChar == '}')
            {
                inVariable = false;
            }
            
            if (currentChar != '$')
            {
                previousCharWasDollar = false;
            }
        }
        
        // End of string
        res.add(path.substring(start, path.length()));
        
        return new PathWrapper(res);
    }
    
    /*
     * In:
     *      "a_${meta1/meta2/meta3}_m_${meta4}_z"
     * Will fill variableNames with:
     *      ["meta1/meta2/meta3", "meta4"]
     */
    private void _fillVariableNames(Collection<ContentType> contentTypes, String element, Set<String> variableNames) throws IllegalArgumentException
    {
        Matcher m = __VARIABLE_REGEXP_PATTERN.matcher(element);
        while (m.find())
        {
            String variableName = m.group(1);
            if (ModelHelper.hasModelItem(variableName, contentTypes) && ModelHelper.getModelItem(variableName, contentTypes) instanceof ElementDefinition)
            {
                variableNames.add(variableName);
            }
            else
            {
                throw new IllegalArgumentException("The variable named '" + variableName + "' can not be used in the extraction result path. It is not an attribute of the defined content types");
            }
        }
    }
    
    private Collection<ContentType> _getFirstLevelContentTypes(Extraction extraction)
    {
        return extraction.getExtractionComponents().stream()
                .map(this::_getContentTypeIds)
                .flatMap(Collection::stream)
                .map(_contentTypeExtensionPoint::getExtension)
                .collect(Collectors.toList());
    }
    
    private Collection<String> _getContentTypeIds(ExtractionComponent component)
    {
        if (component instanceof AbstractSolrExtractionComponent)
        {
            String queryReferenceId = ((AbstractSolrExtractionComponent) component).getQueryReferenceId();
            if (StringUtils.isNotEmpty(queryReferenceId))
            {
                return _editExtractionNodeManager.getSavedQueryContentTypes(queryReferenceId);
            }
        }
        return component.getContentTypes();
    }
    
    /*
     * Out:
     *      A map with the resolved relative paths for each content
     */
    private Map<Content, Set<PathWrapper>> _pathByContent(PathWrapper unresolvedPath, List<Content> contents)
    {
        Map<Content, Set<PathWrapper>> pathByContent = new HashMap<>();
        for (Content content : contents)
        {
            List<Set<String>> pathElements = _resolvePath(unresolvedPath, content);
            Set<PathWrapper> allPaths = _getAllPaths(pathElements);
            pathByContent.put(content, allPaths);
        }
        return pathByContent;
    }
    
    /*
     * Out:
     *      The (resolved) relative paths (as a list of possible elements in a set) for the given content
     */
    private List<Set<String>> _resolvePath(PathWrapper unresolvedPath, Content content)
    {
        List<Set<String>> resolvedPathElements = new ArrayList<>();
        for (String element : unresolvedPath.getElements())
        {
            Set<String> resolvedElements = _resolvePathElement(element, content);
            resolvedPathElements.add(_validPathElementNames(resolvedElements));
        }
        
        return resolvedPathElements;
    }
    
    /*
     * Out:
     *      The (resolved) possible path elements (i.e. folder names) for the given values (i.e. variables resolved for a given content)
     *      It is a set as variables can be multivalued
     */
    private Set<String> _resolvePathElement(String unresolvedElement, Content content)
    {
        Map<String, Set<String>> replacements = new HashMap<>();
        Matcher m = __VARIABLE_REGEXP_PATTERN.matcher(unresolvedElement);
        while (m.find())
        {
            String variableName = m.group(1);
            ElementType type = content.getType(variableName);
            Object variableValue = content.getValue(variableName, true);
            Set<String> strValues = _getStringValues(type, variableValue);
            replacements.put("${" + variableName + "}", strValues);
        }
        
        Set<String> pathElements = Collections.singleton(unresolvedElement);
        for (String toReplace : replacements.keySet())
        {
            pathElements = _replace(toReplace, replacements.get(toReplace), pathElements);
        }
        return pathElements;
    }
    
    @SuppressWarnings("unchecked")
    private Set<String> _getStringValues(ElementType type, Object value)
    {
        Stream<Object> values = Stream.empty();
        if (type.getManagedClassArray().isInstance(value))
        {
            values = Arrays.stream((Object[]) value);
        }
        else
        {
            values = Collections.singleton(value).stream();
        }
        
        Set<String> strValues = values.filter(Objects::nonNull)
                                      .map(type::toString)
                                      .collect(Collectors.toSet());
        
        if (strValues.isEmpty())
        {
            strValues = Collections.singleton(__NO_VALUE_OR_BLANK_FOLDER_NAME);
        }
        return strValues;
    }
    
    /*
     * In:
     *      toReplace="${metaB}"
     *      replaceBy={ "b1", "b2" }
     *      uncompleteElements={ "a1_${metaB}_${metaC}", "a2_${metaB}_${metaC}" }
     * Out:
     *      { "a1_b1_${metaC}", "a2_b1_${metaC}", "a1_b2_${metaC}", "a2_b2_${metaC}" }
     */
    private Set<String> _replace(String toReplace, Set<String> replaceBy, Set<String> uncompleteElements)
    {
        Set<String> newPossibleElements = new HashSet<>();
        for (String singleReplaceBy : replaceBy)
        {
            for (String uncompleteElement : uncompleteElements)
            {
                newPossibleElements.add(uncompleteElement.replace(toReplace, singleReplaceBy));
            }
        }
        return newPossibleElements;
    }
    
    private Set<String> _validPathElementNames(Set<String> elements)
    {
        return elements.stream()
                .map(this::_validPathElementName)
                .collect(Collectors.toSet());
    }
    
    /*
     * Out:
     *      The tranformed path element name to have a valid folder name
     */
    private String _validPathElementName(String element)
    {
        return StringUtils.isBlank(element) ? __NO_VALUE_OR_BLANK_FOLDER_NAME : FilenameUtils.filterName(element);
    }
    
    /*
     * In:
     *      [{a1, a2}, {b}, {c1, c2}]
     * Out:
     *      {[a1, b c1], [a1, b, c2], [a2, b, c1], [a2, b, c2]}
     *      representing {a1/b/c1, a1/b/c2, a2/b/c1, a2/b/c2}
     */
    private Set<PathWrapper> _getAllPaths(List<Set<String>> pathElements)
    {
        Set<PathWrapper> allPaths = new HashSet<>();
        allPaths.add(null); // root
        for (Set<String> possibleElements : pathElements)
        {
            allPaths = _getAllPathsInCurrentLevel(possibleElements, allPaths);
        }
        return allPaths;
    }
    
    private Set<PathWrapper> _getAllPathsInCurrentLevel(Set<String> possibleElementsInCurrentLevel, Set<PathWrapper> computedPathsInPreviousLevel)
    {
        Set<PathWrapper> paths = new HashSet<>();
        for (PathWrapper computedPathInPreviousLevel : computedPathsInPreviousLevel)
        {
            for (String possibleElement : possibleElementsInCurrentLevel)
            {
                List<String> pathInCurrentLevel;
                if (computedPathInPreviousLevel == null) // root case
                {
                    pathInCurrentLevel = new ArrayList<>();
                }
                else
                {
                    pathInCurrentLevel = new ArrayList<>(computedPathInPreviousLevel.getElements());
                }
                pathInCurrentLevel.add(possibleElement);
                paths.add(new PathWrapper(pathInCurrentLevel));
            }
        }
        return paths;
    }
    
    /*
     * In:
     *      A map with the resolved relative paths for each content (the different possible paths are within a set)
     * Out:
     *      The 'inverted' map, i.e. a map with the list of contents for each path
     */
    private Map<PathWrapper, List<Content>> _contentsByPath(Map<Content, Set<PathWrapper>> pathByContent)
    {
        Map<PathWrapper, List<Content>> contentsByPath = new HashMap<>();
        for (Content content : pathByContent.keySet())
        {
            Set<PathWrapper> paths = pathByContent.get(content);
            for (PathWrapper path : paths)
            {
                List<Content> contentsForPath;
                if (contentsByPath.containsKey(path))
                {
                    contentsForPath = contentsByPath.get(path);
                }
                else
                {
                    contentsForPath = new ArrayList<>();
                    contentsByPath.put(path, contentsForPath);
                }
                contentsForPath.add(content);
            }
        }
        return contentsByPath;
    }
    
    // Just for readability of the code (PathWrapper in method signatures is better than List<String>)
    private static final class PathWrapper
    {
        private List<String> _pathElements;

        PathWrapper(List<String> pathElements)
        {
            _pathElements = pathElements;
        }
        
        List<String> getElements()
        {
            return _pathElements;
        }

        @Override
        public int hashCode()
        {
            final int prime = 31;
            int result = 1;
            result = prime * result + ((_pathElements == null) ? 0 : _pathElements.hashCode());
            return result;
        }

        @Override
        public boolean equals(Object obj)
        {
            if (this == obj)
            {
                return true;
            }
            if (obj == null)
            {
                return false;
            }
            if (!(obj instanceof PathWrapper))
            {
                return false;
            }
            PathWrapper other = (PathWrapper) obj;
            if (_pathElements == null)
            {
                if (other._pathElements != null)
                {
                    return false;
                }
            }
            else if (!_pathElements.equals(other._pathElements))
            {
                return false;
            }
            return true;
        }
        
        @Override
        public String toString()
        {
            return _pathElements.toString();
        }
    }
}
