/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution.pipeline;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.AbstractGenerator;
import org.apache.cocoon.serialization.ZipArchiveSerializer;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceException;
import org.apache.excalibur.source.impl.FileSource;
import org.xml.sax.SAXException;

/**
 * Generate the list of file to include in a zip for downloading results files 
 */
public class ExtractionResultsDownloadGenerator extends AbstractGenerator
{
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String[] filePaths = request.getParameterValues("file");
        
        contentHandler.startDocument();
        contentHandler.startPrefixMapping("zip", ZipArchiveSerializer.ZIP_NAMESPACE);
        XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");
        
        Set<String> saxedPath = new HashSet<>();
        for (String filePath : filePaths)
        {
            FileSource fileSource = (FileSource) resolver.resolveURI("ametys-home://data/extraction/" + filePath);
            String parentPath = filePath.contains("/") ? filePath.substring(0, filePath.lastIndexOf("/")) : StringUtils.EMPTY;
            saxedPath.addAll(_resultFileSourceToSAX(fileSource, parentPath, saxedPath));
        }
        
        XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "archive");
        contentHandler.endPrefixMapping("zip");
        contentHandler.endDocument();
    }
    
    /**
     * Generate SAX events for the given result file source
     * @param resultSource the result file source
     * @param parentPath the path of the parent folder
     * @param alreadySaxedPaths paths of the files that have already been saxed
     * @return The paths of the files that have been saxed
     * @throws SourceException if an error occurs while traversing a collection source
     * @throws SAXException if an error occurs while generating SAX events
     */
    protected Set<String> _resultFileSourceToSAX(FileSource resultSource, String parentPath, Set<String> alreadySaxedPaths) throws SourceException, SAXException
    {
        Set<String> allSaxedPaths = new HashSet<>(alreadySaxedPaths);
        String currentSourcePath = StringUtils.isEmpty(parentPath) ? resultSource.getName() : parentPath + "/" + resultSource.getName();
        
        if (resultSource.isCollection())
        {
            Collection<FileSource> children = resultSource.getChildren();
            for (FileSource child : children)
            {
                allSaxedPaths.addAll(_resultFileSourceToSAX(child, currentSourcePath, allSaxedPaths));
            }
        }
        else if (allSaxedPaths.add(currentSourcePath))
        {
            AttributesImpl zipAttrs = new AttributesImpl();
            zipAttrs.addAttribute("", "name", "name", "CDATA", currentSourcePath);
            zipAttrs.addAttribute("", "src", "src", "CDATA", resultSource.getURI());
            XMLUtils.startElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry", zipAttrs);
            XMLUtils.endElement(contentHandler, ZipArchiveSerializer.ZIP_NAMESPACE, "entry");
        }
        
        return allSaxedPaths;
    }
}
