/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution.pipeline;

import java.io.OutputStream;
import java.util.List;
import java.util.Map;

import org.ametys.plugins.extraction.execution.ExtractionExecutor;
import org.ametys.plugins.extraction.execution.PathResolver;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * A descriptor which can provide a {@link Pipeline} with {@link #newPipeline(OutputStream) newPipeline} method.
 */
public interface PipelineDescriptor
{
    /**
     * Gets the label
     * @return the label
     */
    I18nizableText getLabel();
    
    /**
     * Gets the {@link ExtractionMatcher extraction matcher} to check if the pipeline is able to handle a given extraction
     * @return the {@link ExtractionMatcher extraction matcher} of the pipeline
     */
    ExtractionMatcher getExtractionMatcher();
    
    /**
     * Gets the XSLT file names to chain
     * @return the XSLT file names to chain
     */
    List<String> getStylesheets();
    
    /**
     * Gets the {@link PipelineSerializerModel serializer model}
     * @return the {@link PipelineSerializerModel serializer model}
     */
    PipelineSerializerModel getSerializerModel();
    
    /**
     * Gets the result subfolder (or file). 
     * <br>Can contain variables (see {@link PathResolver}), resolved against the extraction results.
     * <br>If it returns
     * <ul>
     * <li>a file path without variable, then all extraction results will be in this file;</li>
     * <li>a file path with variables, then extraction results will be dispatched in the resolved file paths;</li>
     * <li>
     * a folder path without variable, then all extraction results will be in one file under this folder 
     * (the file name will be the one provided in <code>defaultResultFileName</code> parameter 
     * of {@link ExtractionExecutor#execute(String, String, String, java.util.Map, PipelineDescriptor) ExtractionExecutor#execute method});
     * </li>
     * <li>
     * a folder path with variables, then each first level content extraction result will be dispatched in one file under the resolved folders
     * (the file name will be the <b>title</b> of the content, and the extension is the return of {@link #getDefaultExtension()});
     * </li>
     * </ul>
     * @return the result subfolder (or file)
     */
    String getResultSubfolder();
    
    /**
     * Gets the default extension of the file result(s)
     * <br>The default extension is used when {@link #getResultSubfolder()} is a folder and not a file.
     * @return the default extension of the file result(s)
     */
    String getDefaultExtension();
    
    /**
     * Provides a new {@link Pipeline}
     * @param out the output stream to write on
     * @return the built {@link Pipeline}
     */
    Pipeline newPipeline(OutputStream out);
    
    /**
     * Get the output parameters to add some specific features in each serializer.
     * @return The output parameters
     */
    Map<String, String> getOutputParameters();
}
