/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.execution.pipeline;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;

import javax.xml.transform.TransformerFactory;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.ametys.core.util.LambdaUtils;

/**
 * Convenient method for {@link Pipeline} API implementations
 */
public final class Pipelines
{
    private static SAXTransformerFactory __transformerFactory;
    
    private Pipelines()
    {
        // Nothing
    }
    
    /**
     * Gets a {@link SAXTransformerFactory}
     * @return a {@link SAXTransformerFactory}
     */
    public static SAXTransformerFactory getSaxTransformerFactory()
    {
        if (__transformerFactory == null)
        {
            __transformerFactory = (SAXTransformerFactory) TransformerFactory.newInstance();
        }
        return __transformerFactory;
    }
    
    /**
     * Gets the output stream of the given file (which will be created if it does not exist).
     * @param outputFile The output file. Will be created if needed.
     * @return the output stream
     * @throws IOException if an I/O error occured
     */
    public static OutputStream getOutputStream(Path outputFile) throws IOException
    {
        outputFile.getParent().toFile().mkdirs();
        OutputStream resultOs = Files.newOutputStream(outputFile);
        return resultOs;
    }
    
    /**
     * Sets the result of a transformer handler into an output stream.
     * @param transformerHandler The transformer handler
     * @param outputStream The output stream
     * @return the stream result
     * @throws IOException if an I/O error occured
     */
    public static StreamResult setResult(TransformerHandler transformerHandler, OutputStream outputStream) throws IOException
    {
        StreamResult streamResult = new StreamResult(outputStream);
        transformerHandler.setResult(streamResult);
        return streamResult;
    }
    
    /**
     * Close the given stream result
     * @param result the stream result
     */
    public static void close(StreamResult result)
    {
        Optional.ofNullable(result)
                .map(StreamResult::getOutputStream)
                .ifPresent(LambdaUtils.wrapConsumer(OutputStream::close));
    }
}
