/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.extraction.rights;

import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.RightsException;
import org.ametys.plugins.core.impl.right.StringHierarchicalAccessController;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * AccessController handling access to extraction
 */
public class ExtractionAccessController extends StringHierarchicalAccessController
{
    /** the extraction context category */
    public static final I18nizableText EXTRACTION_CONTEXT_CATEGORY = new I18nizableText("plugin.extraction", "RIBBON_TABS_TAB_EXTRACTION_LABEL");
    /** The prefix for rights on the root of a collection */
    public static final String ROOT_CONTEXT = "/extraction-dir";

    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object)
    {
        if (object instanceof String str && StringUtils.startsWith(str, ExtractionAccessController.ROOT_CONTEXT))
        {
            if (StringUtils.equals(str, ExtractionAccessController.ROOT_CONTEXT))
            {
                return new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_ACCESS_CONTROLLER_ROOT_EXPLANATION_LABEL");
            }
            else
            {
                Map<String, I18nizableTextParameter> params = Map.of("contextLabel", getObjectLabel(object));
                String target = StringUtils.substringAfterLast(str, SEPARATOR);
                if (StringUtils.endsWith(target, ".xml"))
                {
                    return new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_ACCESS_CONTROLLER_FILE_EXPLANATION_LABEL", params);
                }
                else
                {
                    return new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_ACCESS_CONTROLLER_FOLDER_EXPLANATION_LABEL", params);
                }
            }
        }
        return super.getObjectLabel(object);
    }
    
    @Override
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof String str && StringUtils.startsWith(str, ExtractionAccessController.ROOT_CONTEXT))
        {
            if (StringUtils.equals(str, ExtractionAccessController.ROOT_CONTEXT))
            {
                return new I18nizableText("plugin.extraction", "PLUGINS_EXTRACTION_ACCESS_CONTROLLER_ROOT_LABEL");
            }
            else
            {
                String target = StringUtils.substringAfter(str, ROOT_CONTEXT);
                target = StringUtils.replace(target.substring(1), "/", " > ");
                return new I18nizableText(target);
            }
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        return EXTRACTION_CONTEXT_CATEGORY;
    }
    
    public int getObjectPriority(Object object)
    {
        if (object instanceof String str && StringUtils.equals(str, ExtractionAccessController.ROOT_CONTEXT))
        {
            return 10;
        }
        return super.getObjectPriority(object);
    }
}
