/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebHelper;

/**
 * Remove the file stored by the flipbook cache in ametys home when a content is deleted or modified.
 * The observer target the file created by {@link ConvertContentAttachment2ImagesComponent} and {@link ConvertMetadata2ImagesComponent}.
 * 
 * In case of deletion, the cache could store outdated attachments or metadata.
 * In case of modification, the cache could store outdated metadata.
 * The modification of the attachment is handled by {@link CleanFlipbookCacheOnResourceDeletedObserver}
 */
public class CleanFlipbookCacheOnContentDeletedObserver extends AbstractLogEnabled implements Observer, Serviceable, Contextualizable
{

    private AmetysObjectResolver _resolver;
    private ConvertMetadata2ImagesComponent _attributeImageComponent;
    private Context _context;
    private ConvertContentAttachment2ImagesComponent _attachmentImageComponent;

    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _attributeImageComponent = (ConvertMetadata2ImagesComponent) manager.lookup(ConvertMetadata2ImagesComponent.ROLE);
        _attachmentImageComponent = (ConvertContentAttachment2ImagesComponent) manager.lookup(ConvertContentAttachment2ImagesComponent.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    public boolean supports(Event event)
    {
        return ObservationConstants.EVENT_CONTENT_DELETING.equals(event.getId())
            || ObservationConstants.EVENT_CONTENT_MODIFIED.equals(event.getId());
    }

    public int getPriority()
    {
        return Observer.MIN_PRIORITY;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Map<String, Object> arguments = event.getArguments();
        String contentId = (String) arguments.get(ObservationConstants.ARGS_CONTENT_ID);
        Content content = _resolver.resolveById(contentId);
        
        Request request = ContextHelper.getRequest(_context);
        String siteName = WebHelper.getSiteName(request, content);
        
        _attributeImageComponent.doCleanCache(content, siteName);
        
        // only delete the attachment cache if the content was deleted
        if (ObservationConstants.EVENT_CONTENT_DELETING.equals(event.getId()))
        {
            _attachmentImageComponent.doCleanCache(content, siteName);
        }
    }

}
