/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.io.InputStream;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.codec.digest.DigestUtils;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.FilenameUtils;
import org.ametys.plugins.explorer.resources.Resource;

/**
 * Implementation of a {@link AbstractConvertDocument2ImagesComponent} for a content attachment
 */
public class ConvertContentAttachment2ImagesComponent extends AbstractConvertDocument2ImagesComponent
{
    /** Avalon ROLE. */
    public static final String ROLE = ConvertContentAttachment2ImagesComponent.class.getName();
    
    // match a resource stored as a content attachment
    private static final Pattern __ATTACHMENTS_PATTERN = Pattern.compile("^.*/ametys-internal:sites/([^/]+)/ametys-internal:contents/([^/]+)/ametys-internal:attachments/(.*)$");
    
    /**
     * Put the file in cache
     * @param resource the resource
     * @param contentName the name of the content
     * @param siteName the name of the site
     * @return The absolute cache path
     * @throws Exception if an error occurs while caching the file
     */
    public String doCache(Resource resource, String contentName, String siteName) throws Exception
    {
        // Ensure the base folder exists.
        String cachePath = _getCacheDirectory (resource.getResourcePath(), contentName, siteName);
        
        try (InputStream is = resource.getInputStream(); InputStream is2 = resource.getInputStream())
        {
            String md5sum = DigestUtils.md5Hex(is);
            return cache(cachePath, md5sum, is2, resource.getName(), resource.getId(), resource.getMimeType());
        }
    }
    
    private String _getCacheDirectory(String resourcePath, String contentName, String siteName)
    {
        // Content attachments.
        StringBuilder buff = _getContentAttachmentsCacheDirectory(contentName, siteName);
        buff.append(FilenameUtils.encodePath(resourcePath));
        
        return buff.toString();
    }
    
    private StringBuilder _getContentAttachmentsCacheDirectory(String contentName, String siteName)
    {
        StringBuilder buff = new StringBuilder();
        
        buff.append("/");
        buff.append(siteName);
        buff.append("/contents/");
        buff.append(contentName);
        buff.append("/attachments");
        
        return buff;
    }

    /**
     * Remove content attachments image stored in the flipbook cache for the given content and sitename
     * @param content the linked content
     * @param siteName the sitename
     */
    public void doCleanCache(Content content, String siteName)
    {
        cleanCache(_getContentAttachmentsCacheDirectory(content.getName(), siteName).toString());
    }

    /**
     * Remove the image stored in cache that are associated with the resource path.
     * If the provided resource path doesn't point to a content attachment, calling this method will have no effect.
     * @param path the JCR path of the resource to clean
     */
    public void doCleanCache(String path)
    {
        Matcher attachmentMatcher = __ATTACHMENTS_PATTERN.matcher(path);

        if (attachmentMatcher.matches())
        {
            cleanCache(_getCacheDirectory("/" + attachmentMatcher.group(3), attachmentMatcher.group(2), attachmentMatcher.group(1)));
        }
    }
}
