/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.io.InputStream;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.codec.digest.DigestUtils;

import org.ametys.cms.data.Binary;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.FilenameUtils;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.type.ModelItemTypeConstants;
import org.ametys.runtime.model.ModelItem;

/**
 * Implementation of a {@link AbstractConvertDocument2ImagesComponent} for a binary metadata
 */
public class ConvertMetadata2ImagesComponent extends AbstractConvertDocument2ImagesComponent
{
    /** Avalon ROLE. */
    public static final String ROLE = ConvertMetadata2ImagesComponent.class.getName();
    
    /**
     * Put the file in cache
     * @param content the content
     * @param metadataPath the path of the metadata
     * @param siteName the name of the site
     * @return The absolute cache path
     * @throws Exception if an error occurs while caching the file
     */
    public String doCache(Content content, String metadataPath, String siteName) throws Exception
    {
        Binary binary = _getBinary(metadataPath, content);

        String cachePath = _getCacheDirectory(metadataPath, binary.getFilename(), content.getName(), siteName);
        try (InputStream is = binary.getInputStream(); InputStream is2 = binary.getInputStream())
        {
            String md5sum = DigestUtils.md5Hex(is);
            return cache(cachePath, md5sum, is2, content.getName(), content.getId(), binary.getMimeType());
        }
    }
    
    private Binary _getBinary(String dataPath, Content content)
    {
        if (dataPath.contains("["))
        {
            // The path contains a repeater entry with the new syntax ("repeater[1]/myBinaryData"), the getValue method can be used
            return content.getValue(dataPath);
        }
        else
        {
            // FILPBOOK-63: The path could contain a repeater with the old syntax ("repeater/1/myBinaryData"), look over each path segment to get the binary data
            // TODO NEWATTRIBUTEAPI: When all generators have been migrated to the new API, this code is unnecessary
            List<String> pathSegments = Arrays.asList(dataPath.split(ModelItem.ITEM_PATH_SEPARATOR));
            Iterator<String> pathSegmentsIt = pathSegments.iterator();
            
            ModelAwareDataHolder dataHolder = content;
            Binary binary = null;
            while (pathSegmentsIt.hasNext())
            {
                String pathSegment = pathSegmentsIt.next();
                
                if (pathSegmentsIt.hasNext())
                {
                    // Not the last segment: it is a repeater or a composite
                    if (dataHolder.getType(pathSegment).getId().equals(ModelItemTypeConstants.REPEATER_TYPE_ID))
                    {
                        ModelAwareRepeater repeater = dataHolder.getRepeater(pathSegment);
                        int repeaterPosition = Integer.valueOf(pathSegmentsIt.next());
                        dataHolder = repeater.getEntry(repeaterPosition);
                    }
                    else
                    {
                        dataHolder = dataHolder.getComposite(pathSegment);
                    }
                }
                else
                {
                    binary = dataHolder.getValue(pathSegment);
                }
            }
            
            return binary;
        }
    }
    
    private String _getCacheDirectory(String metadataPath, String fileName, String contentName, String siteName)
    {
        StringBuilder buff = _getContentCacheDirectory(contentName, siteName);
        buff.append("/metadatas/");
        buff.append(metadataPath);
        buff.append("/");
        buff.append(FilenameUtils.encodeName(fileName));
        
        return buff.toString();
    }

    private StringBuilder _getContentCacheDirectory(String contentName, String siteName)
    {
        StringBuilder buff = new StringBuilder();
        buff.append("/");
        buff.append(siteName);
        buff.append("/contents/");
        buff.append(contentName);
        return buff;
    }
    
    /**
     * Remove all cached attribute image for the given content and sitename in the flipbook cache
     * @param content the content linked to the info to remove
     * @param siteName the sitename linked to the info to remove
     */
    public void doCleanCache(Content content, String siteName)
    {
        // target the metadata cache only to avoid removing the attachment cache if its unnecessary
        cleanCache(_getContentCacheDirectory(content.getName(), siteName).toString() + "/metadatas");
    }
}
