/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The observer target the file created by {@link ConvertResource2ImagesComponent} and {@link ConvertExternalResource2ImagesComponent} and {@link ConvertContentAttachment2ImagesComponent}
 * 
 * A flipbook resource can be stored in the site resource, the shared resource or an attachment
 */
public class CleanFlipbookCacheOnResourceDeletedObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    private ConvertResource2ImagesComponent _resourceImageComponent;
    private ConvertContentAttachment2ImagesComponent _attachmentImageComponent;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resourceImageComponent = (ConvertResource2ImagesComponent) manager.lookup(ConvertResource2ImagesComponent.ROLE);
        _attachmentImageComponent = (ConvertContentAttachment2ImagesComponent) manager.lookup(ConvertContentAttachment2ImagesComponent.ROLE);
    }
    
    public boolean supports(Event event)
    {
        // handle the deletion of a resource or a collection containing a resource
        return ObservationConstants.EVENT_RESOURCE_DELETED.equals(event.getId())
            || ObservationConstants.EVENT_COLLECTION_DELETED.equals(event.getId())
            || ObservationConstants.EVENT_RESOURCE_MOVED.equals(event.getId())
            || ObservationConstants.EVENT_COLLECTION_MOVED.equals(event.getId())
            || ObservationConstants.EVENT_RESOURCE_RENAMED.equals(event.getId())
            || ObservationConstants.EVENT_COLLECTION_RENAMED.equals(event.getId());
    }

    public int getPriority()
    {
        return Observer.MIN_PRIORITY;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Map<String, Object> args = event.getArguments();
        String path = (String) args.get(ObservationConstants.ARGS_PATH);
        String resourcePath = (String) args.get(ObservationConstants.ARGS_RESOURCE_PATH);
        if (resourcePath == null)
        {
            resourcePath = (String) args.get(ObservationConstants.ARGS_EXPLORER_PATH);
        }
        if (ObservationConstants.EVENT_RESOURCE_MOVED.equals(event.getId())
            || ObservationConstants.EVENT_COLLECTION_MOVED.equals(event.getId())
            || ObservationConstants.EVENT_RESOURCE_RENAMED.equals(event.getId())
            || ObservationConstants.EVENT_COLLECTION_RENAMED.equals(event.getId()))
        {
            path = (String) args.get("object.old.path");
            resourcePath = (String) args.get("resource.old.path");
            if (resourcePath == null)
            {
                resourcePath = (String) args.get("explorer.old.path");
            }
        }
        
        _resourceImageComponent.doCleanCache(path, resourcePath);
        _attachmentImageComponent.doCleanCache(path);
    }
}
