/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.flipbook;

import java.io.InputStream;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.codec.digest.DigestUtils;

import org.ametys.core.util.FilenameUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.web.explorer.ResourceHelper;

/**
 * Implementation of a {@link AbstractConvertDocument2ImagesComponent} for a {@link Resource}
 */
public class ConvertResource2ImagesComponent extends AbstractConvertDocument2ImagesComponent
{   
    /** Avalon ROLE. */
    public static final String ROLE = ConvertResource2ImagesComponent.class.getName();
    
    // match a resource in site
    private static final Pattern __RESOURCE_PATTERN = Pattern.compile("^.*/ametys-internal:sites/([^/]+)/ametys-internal:resources/(.*)$");
    // match a resource in site but with hashed site
    private static final Pattern __ROOT_SITE_RESOURCE_PATTERN = Pattern.compile("^.*/ametys-internal:sites/[^/]+/[^/]+/([^/]+)/ametys-internal:resources/(.*)$");
    // match a resource shared
    private static final Pattern __SHARED_RESOURCE_PATTERN = Pattern.compile("^.*/ametys:plugins/web-explorer/shared-resources/(.*)$");

    /**
     * Put the file in cache
     * @param resource the resource to cache
     * @param siteName the name of the site
     * @return The absolute cache path
     * @throws Exception if an error occurs while caching the file
     */
    public String doCache(Resource resource, String siteName) throws Exception
    {
        // Ensure the base folder exists.
        String cachePath = _getCacheDirectory(resource, siteName);
        
        try (InputStream is = resource.getInputStream(); InputStream is2 = resource.getInputStream())
        {
            String md5sum = DigestUtils.md5Hex(is);
            return cache(cachePath, md5sum, is2, resource.getName(), resource.getId(), resource.getMimeType());
        }
    }
    
    private String _getCacheDirectory(Resource resource, String siteName)
    {
        
        String fullPath = resource.getPath();
        if (fullPath.startsWith(ResourceHelper.SHARED_RESOURCE_PATH))
        {
            // Shared resources.
            return "/shared-resources" + FilenameUtils.encodePath(fullPath.substring(ResourceHelper.SHARED_RESOURCE_PATH.length()));
        }

        StringBuilder buff = new StringBuilder();

        // Non-shared resources.
        buff.append("/");
        buff.append(siteName);
        buff.append("/resources");
        
        buff.append(FilenameUtils.encodePath(resource.getResourcePath()));
        
        return buff.toString();
    }

    /**
     * Remove a resource from the flipbook cache.
     * This method will first try to determine if the provided path either point to shared resource or a site resource.
     * If it's neither, the method won't do anything as this component only handle resource
     * @param path the JCR path of the resource
     * @param resourcePath the resource path (as return by {@link Resource}:getResourcePath)
     */
    public void doCleanCache(String path, String resourcePath)
    {
        Matcher resourceMatcher = __RESOURCE_PATTERN.matcher(path);
        Matcher rootSiteResourceMatcher = __ROOT_SITE_RESOURCE_PATTERN.matcher(path);
        Matcher sharedMatcher = __SHARED_RESOURCE_PATTERN.matcher(path);
        
        String cachePath = null;
        if (resourceMatcher.matches())
        {
            // call ResourceImageComponent
            cachePath = "/" + resourceMatcher.group(1) + "/resources" + FilenameUtils.encodePath(resourcePath);
        }
        else if (rootSiteResourceMatcher.matches())
        {
            // call ResourceImageComponent
            cachePath = "/" + rootSiteResourceMatcher.group(1) + "/resources" + FilenameUtils.encodePath(resourcePath);
        }
        else if (sharedMatcher.matches())
        {
            // call ResourceImageComponent
            cachePath = "/shared-resources" + FilenameUtils.encodePath(path.substring(ResourceHelper.SHARED_RESOURCE_PATH.length()));
        }
        if (cachePath != null)
        {
            cleanCache(cachePath);
        }
    }
}
